/**
 * (c)2013 Tomasz Francuz
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 *    Atmel microcontroller product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <avr/interrupt.h>
#include <asf.h>
#include "conf_usb.h"
#include <SPI.h>
#include <Alloc_safe.h>
#include <AT45CMD.h>
#include <DataFLASH.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <string.h>
#include "pf/pff.h"
#include "DataFLASH.h"

//Makro umieszczajce zadany acuch w przestrzeni adresowej __flash
#define PGM_STR(X) ((const __flash char[]) { X })

typedef void (*CmdHandler)(char *param);

typedef struct
{
	const char __flash *cmd;                //Wskanik do polecenia w formacie ASCIIZ
	const CmdHandler  Handler;              //Funkcja realizujca polecenie
} Command;

void Cmd_Dir(char *param);     //Wywietla spis plikw
void Cmd_Write(char *param);   //(nazwa, size) - zapisuje nowy plik
void Cmd_Read(char *param);    //(nazwa, size) - odczytuje plik
void Cmd_Dump(char *param);    //(addr, size) - -odczytuje wskazan liczb bajtw poczwszy od addr

//Lista rozpoznawanych polece
const __flash Command __flash Polecenia[]={{PGM_STR("dir"), Cmd_Dir}, {PGM_STR("write"), Cmd_Write},
	                                       {PGM_STR("read"), Cmd_Read}, {PGM_STR("dump"), Cmd_Dump}};

FATFS fatfs;

int main(void)
{
	void InterpretCommand(char *cmdline)
	{
		uint8_t indeks;
		char *cmd=strtok(cmdline, " \r\n"); //Wydziel polecenie z przekazanej linii
		uint8_t max_indeks=sizeof(Polecenia)/sizeof(Polecenia[0]);
		for(indeks=0; indeks<max_indeks; indeks++)
		{
			if(strcmp_P(cmd, Polecenia[indeks].cmd)==0) //Przeszukaj list polece
			{
				Polecenia[indeks].Handler(cmdline);   //Wywoaj funkcj obsugi przesanego polecenia
				break;
			}
		}
		if(indeks == max_indeks)  //Jeli polecenie nieznane...
		{
			udi_cdc_write_buf(cmd, strlen(cmd));
			udi_cdc_write_buf(" - nieznane polecenie\r\n", 22); //Bd
		}
	}

	SPI_init();            //Zainicjuj interfejs SPI uywany przez pami i transakcyjny system jej obsugi
	DataFLASH_CS_init();   //Zainicjuj kontrol pamici

	PMIC.CTRL = PMIC_LOLVLEN_bm | PMIC_MEDLVLEN_bm | PMIC_HILVLEN_bm; //Wczamy wszystkie poziomy przerwa (potrzebujemy tylko najniszy)
	sei();

	pf_mount(&fatfs);  //Dostp do systemu plikw

	sysclk_init();
	udc_start();

	while (1)
	{
		char Bufor[256];   //Bufor na odebranie linii
		uint8_t indeks=0;

		while(1)
		 {
			if(udi_cdc_is_rx_ready())
			{
				Bufor[indeks] = udi_cdc_getc();
				if(Bufor[indeks] == '\n')
				{
					if((indeks >= 1) && (Bufor[indeks-1] == '\r')) --indeks;
					Bufor[indeks]=0; //Wpisz kocowy znak NUL
					break;           //Odebrano pen lini
				}
				++indeks;
			}
		 }
		InterpretCommand(Bufor);  //Zdekoduj przesane polecenie
	}
}

void main_suspend_action(void)
{
//Interfejs wchodzi w tryb powerdown
}

void main_resume_action(void)
{
//Wybudzenie interfejsu
}

void main_sof_action(void)
{
//	Odebrano ramk
}

bool main_cdc_enable(uint8_t port)
{
	// Otwarto port CDC
	return true;
}

void main_cdc_disable(uint8_t port)
{
	// Zamknito port CDC
}

void main_cdc_set_dtr(uint8_t port, bool b_enable)
{
	if (b_enable) {
		// Host otwar port COM
	}else{
		// Host zamkn port COM
	}
}

void uart_rx_notify(uint8_t port)
{
	//Opcjonalnie powiadomienie o nadejciu znaku
}

void uart_config(uint8_t port, usb_cdc_line_coding_t * cfg)
{
	//Konfiguracja w odpowiedzi na otwarcie CDC
}

void udi_cdc_write_txt(char *txt)
{
	udi_cdc_write_buf(txt, strlen(txt));
}

void Cmd_Dir(char *param)
{
    FRESULT res;
    FILINFO fno;
    DIR dir;
    char *fn;
	char Bufor[27];

	char *name=strtok(NULL, " ,;");             //Pobierz nazw katalogu
    res = pf_opendir(&dir, name);               // Otwrz kattalog
    if (res == FR_OK)
	{
	    for (;;)
		{
		    res = pf_readdir(&dir, &fno);                  //Odczytaj pozycj z katalogu
		    if (res != FR_OK || fno.fname[0] == 0) break;  //Jeli wystpi bd lub nie ma wicej pozycji to zakocz
		    if (fno.fname[0] == '.') continue;             //Jedn z pozycji jest '.' - zignoruj
		    fn = fno.fname;
		    if (fno.fattrib & AM_DIR)
			    sprintf(Bufor, "%-12s <DIR>\r\n", fn);   //Odczytana pozycja jest katalogiem
			   else sprintf(Bufor, "%-12s %u\r\n", fn, fno.fsize); //Odczytana pozycja jest plikiem
			udi_cdc_write_txt(Bufor);           //Wylij nazw przez USB
	    }
    }
}

void Cmd_Write(char *param)
{
	char *name=strtok(NULL, " ,;");             //Pobierz nazw pliku
	__uint24 size=atol(strtok(NULL, " \r\n"));  //Pobierz jego dugo

	if(pf_open(name) == FR_OK)
	{  //Utworzenie nowego pliku powiodo si
		uint8_t indeks=0;
		char Bufor[256];
		WORD zapisane;
		while(size)              //Wczytaj kolejne bajty stanowice dane pliku
		{
			while(udi_cdc_is_rx_ready()==false);  //Zaczekaj na znak
			Bufor[indeks++] = udi_cdc_getc();     //Wpisz znak do bufora
			//Bufor[indeks++] ='c';
			--size;
			if((indeks == 0) || (size == 0))
				pf_write(Bufor, size ? 256 : indeks, &zapisane); //Zapisz dane do pliku
		}
		pf_write(0, 0, &zapisane);   //Zakocz operacj zapisu
		udi_cdc_write_txt("Plik zapisano.\r\n");
	} else udi_cdc_write_txt("Problem z utworzeniem pliku.\r\n");
}

void Cmd_Read(char *param)
{
	char *name=strtok(NULL, " \r\n");          //Pobierz nazw pliku

	if(pf_open(name) == FR_OK)  //Otwrz plik
	{
		uint8_t Bufor[64];
		WORD zapisane;

		do
		{
			pf_read(Bufor, 64, &zapisane);        //Odczytaj porcj danych z pliku
			udi_cdc_write_buf(Bufor, zapisane);   //Wylij dane do PC
		} while(zapisane == 64);
	} else udi_cdc_write_buf("Pliku nie znaleziono.\r\n", 23);
}

void Cmd_Dump(char *param)
{
	uint8_t Bufor[16 + 4 + sizeof(SPI_Transact)];
	char CharBuf[16*4+3];       //Bufor na budowany wiersz
	char chBuf[5];
	SPI_Transact *trans=(SPI_Transact*)&Bufor;
	uint8_t dato, asize;
	param=strtok(NULL, " ,;");
	if(param==NULL) return;      //Bd
	__uint24 start=atol(param);  //Pobierz adres startu
	param=strtok(NULL, " ,;");
	if(param==NULL) return;      //Bd
	__uint24 size=atol(param);   //Ile danych odczyta?

	while(size)
	{
		if(size>16) asize=16; else asize=size;
		dato=DataFlash_AddTransaction(AT45DBX_CMDA_RD_ARRAY_LF_SM, start, Bufor, asize, true); //Odczytaj dane z pamici DataFLASH
		CharBuf[0]=0;
		for(uint8_t i=0; i<asize; i++)   //Zamie odczytane dane na wiersz znakw ASCII
		{
			sprintf(chBuf, "%02X ", trans->data[dato + i]);
			strcat(CharBuf,  chBuf);
		}
		strcat(CharBuf, "\r\n");         //Dodaj znak koca wiersza
		udi_cdc_write_buf(CharBuf, strlen(CharBuf)); //Wylij dane do komputera
		size-=asize;
		start+=asize;
	}
}
