/*
 * SPI.c
 *
 * Created: 2013-04-29 11:29:10
 *  Author: tmf
 */ 


#include "SPI.h"
#include <avr/interrupt.h>
#include <stdbool.h>
#include <util/atomic.h>
#include <avr/io.h>
#include "Alloc_safe.h"

typedef struct
{
	SPI_Transact *elements[SPI_MAXTRANS]; //Wskaniki do transakcji
	uint8_t Beg;                          //Pierwszy element bufora
	uint8_t Count;                        //Liczba elementw w buforze
} CircBuffer;

static CircBuffer SPI_TransBuffer;         //Bufor a transakcje

ISR(DMA_CH1_vect)
{
	static SPI_Transact *trans;            //Wskanik na aktualn transakcj
	DMA_CH1_CTRLB=DMA_CH_ERRIF_bm | DMA_CH_TRNIF_bm | DMA_CH_TRNINTLVL_LO_gc;  //Skasuj flagi przerwania
	if(trans)
	{
		if(trans->SS_Func) trans->SS_Func(false, trans);  //Deaktywuj ukad SPI
		trans->Ready=true;                  //Transakcja zakoczona
		if(trans->SelfDel) free_re(trans);  //Wczone autozwalnianie pamici
	}
	
	SPI_SS_Func_ret SS_ret=SPI_RetOK;
	trans=SPI_cbRead();
	if(trans)                     //Jeli jest oczekujca transakcja...
	{
		if(trans->SS_Func) SS_ret=trans->SS_Func(true, trans);  //Aktywuj ukad SPI
		if(SS_ret==SPI_RetOK)
		{
			DMA_SPI_SetAddr((uint16_t)trans->data, trans->Size); //Uzupenij rejestry adresowe i liczniki DMA
			DMA.CH1.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc; //Odblokuj kana odbiornika
			DMA.CH0.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_1BYTE_gc; //Odblokuj kana nadajnika
		}
	}
}

static inline _Bool SPI_cbIsFull()
{
	return SPI_TransBuffer.Count == SPI_MAXTRANS;
}

static inline _Bool SPI_cbIsEmpty()
{
	return SPI_TransBuffer.Count == 0;
}

_Bool SPI_cbAdd(SPI_Transact *elem)
{
	ATOMIC_BLOCK(ATOMIC_RESTORESTATE)
	{
		if(SPI_cbIsFull()) return false;                   //Czy jest miejsce w kolejce?
		uint8_t end = (SPI_TransBuffer.Beg + SPI_TransBuffer.Count) % SPI_MAXTRANS;
		SPI_TransBuffer.elements[end] = elem;              //Dodaj transakcj
		++SPI_TransBuffer.Count;                           //Liczba elementw w buforze
		if(SPI_TransBuffer.Count == 1) DMA_CH1_vect();
	}
	return true;      //Wszystko ok
}

SPI_Transact *SPI_cbRead()
{
	SPI_Transact *elem;
	ATOMIC_BLOCK(ATOMIC_RESTORESTATE)
	{
		if(SPI_cbIsEmpty()) return NULL;       //Bufor pusty, nie mona zwrci elementu
		elem = SPI_TransBuffer.elements[SPI_TransBuffer.Beg];
		SPI_TransBuffer.Beg = (SPI_TransBuffer.Beg + 1) % SPI_MAXTRANS;
		--SPI_TransBuffer.Count;               //Zmniejszamy liczb elementw pozostaych
	}		                                   //w buforze
	return elem;
}

void DMA_SPI_init()
{
	//Konfigurujemy kana nadajnika
	//Adres docelowy jest stay, rdowy ulega inkrementacji
	DMA.CH0.ADDRCTRL=DMA_CH_SRCRELOAD_NONE_gc | DMA_CH_SRCDIR_INC_gc | DMA_CH_DESTRELOAD_NONE_gc | DMA_CH_DESTDIR_FIXED_gc;
	DMA.CH0.TRIGSRC=DMA_CH_TRIGSRC_USARTD0_DRE_gc;            //Wyzwalanie przez pusty bufor nadajnika
	DMA.CH0.DESTADDR0=(uint16_t)&USARTD0_DATA & 0xff;         //Adres docelowy - rejestr USART_DATA
	DMA.CH0.DESTADDR1=((uint16_t)&USARTD0_DATA >> 8) & 0xff;
	DMA.CH0.DESTADDR2=0;
	//Konfigurujemy kana odbiornika
	//Adres rdowy jest stay, docelowy ulega inkrementacji
	DMA.CH1.ADDRCTRL=DMA_CH_SRCRELOAD_NONE_gc | DMA_CH_SRCDIR_FIXED_gc| DMA_CH_DESTRELOAD_NONE_gc | DMA_CH_DESTDIR_INC_gc;
	DMA.CH1.TRIGSRC=DMA_CH_TRIGSRC_USARTD0_RXC_gc; //Wyzwalanie przez peny bufor odbiornika
	DMA.CH1.SRCADDR0=(uint16_t)&USARTD0_DATA & 0xff;         //Adres docelowy - rejestr USART_DATA
	DMA.CH1.SRCADDR1=((uint16_t)&USARTD0_DATA >> 8) & 0xff;
	DMA.CH1.SRCADDR2=0;
	
	DMA.CH1.CTRLB=DMA_CH_TRNINTLVL_LO_gc;  //Odblokuj przerwanie koca transferu DMA
	DMA.CTRL|=DMA_CH_ENABLE_bm;            //Odblokuj kontroler DMA
}

void SPI_init()
{
	PORTD.DIR|=PIN3_bm | PIN1_bm;            //MOSI, SCK wyjcia
	PORTD.OUTSET=PIN3_bm | PIN1_bm;          //MOSI i SCK w stanie wysokim
	USARTD0.BAUDCTRLA=0;
	USARTD0.BAUDCTRLB=0;                     //Fclk=FPER/2 - maksymalne taktowanie SPI
	USARTD0.CTRLC=USART_CMODE_MSPI_gc;       //Tryb SPI 0
	USARTD0.CTRLB=USART_TXEN_bm | USART_RXEN_bm;
	
	DMA_SPI_init();                          //Wstpnie zainicjuj kontroler DMA
}

//Wpisz adres bufora nadawczego i odbiorczego do DMA
void DMA_SPI_SetAddr(uint16_t addr, uint16_t size)
{
	DMA.CH0.SRCADDR0=addr & 0xff;
	DMA.CH0.SRCADDR1=(addr >> 8) & 0xff;
	DMA.CH0.SRCADDR2=0;
	DMA.CH0.TRFCNT=size;
	DMA.CH0.REPCNT=1;
	DMA.CH1.TRFCNT=size;
	DMA.CH1.REPCNT=1;
	DMA.CH1.DESTADDR0=addr & 0xff;
	DMA.CH1.DESTADDR1=(addr >> 8) & 0xff;
	DMA.CH1.DESTADDR2=0;
}
