/**
 * (c)2013 Tomasz Francuz
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an
 *    Atmel microcontroller product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <avr/interrupt.h>
#include <asf.h>
#include "conf_usb.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "ADC.h"
#include "SPI.h"
#include "AT45IO.h"
#include "AT45CMD.h"
#include "RTC.h"
#include "calendar.h"
#include "ADV/ADVFunc.h"

//Makro umieszczajce zadany acuch w przestrzeni adresowej __flash
#define PGM_STR(X) ((const __flash char[]) { X })

typedef void (*CmdHandler)(char *param);

typedef struct
{
	const char __flash *cmd;                //Wskanik do polecenia w formacie ASCIIZ
	const CmdHandler  Handler;              //Funkcja realizujca polecenie
} Command;

void CMD_date(char *param);                 //Zwr lub ustaw dat moduu
void CMD_erase(char *param);                //Skasuj dane w pamici DataFLASH
void CMD_get(char *param);                  //Zwr dane w formacie csv
void CMD_start(char *param);                //Rozpocznij akwizycj danych
void CMD_stop(char *param);                 //Zakocz akwizycj danych
void CMD_dump(char *param);                 //Odczytaj zawarto DataFLASH jako dane raw

void CMD_Bufdump(char *param);

//Lista rozpoznawanych polece
const __flash Command __flash Polecenia[]={{PGM_STR("date"), CMD_date}, {PGM_STR("erase"), CMD_erase}, {PGM_STR("get"), CMD_get},
										   {PGM_STR("start"), CMD_start}, {PGM_STR("stop"), CMD_stop}, {PGM_STR("dump"), CMD_dump},
										   {PGM_STR("bufdump"), CMD_Bufdump}};

volatile uint8_t Buttons;         //Stan przyciskw - 1 wcinity, 0 - zwolniony

ISR(TCF0_OVF_vect)
{
	static uint8_t counters[2];     //Pomocnicze liczniki

	uint8_t tmpbtn=0;
	tmpbtn=((PORTE_IN & PIN5_bm) == 0);
	tmpbtn|=(((PORTF_IN & PIN1_bm) == 0) << 1);

	uint8_t btnmask=1;
	for(uint8_t i=0; i<2; i++)
	{
		if(counters[i]==0)              //Nie eliminujemy aktualnie drga
		{
			if((tmpbtn ^ Buttons) & btnmask)
			{
				Buttons=(Buttons & (~btnmask)) | (tmpbtn & btnmask); //Przepisz stan klawisza
				counters[i]=10;	//Czas przez jaki stan przycisku bdzie ignorowany
			}
		}
		if(counters[i]) --counters[i];  //Zmniejszaj licznik do 0
		btnmask<<=1;
	}
}

void Kbd_init()
{
	PORTE_DIRCLR=PIN5_bm;  //Waciwie niepotrzebne, gdy domylnie te piny s wejciem
	PORTF_DIRCLR=PIN1_bm;
	TCF0.CTRLB=TC_WGMODE_NORMAL_gc;       //Zwyky tryb pracy timera
	TCF0.INTCTRLA=TC_OVFINTLVL_LO_gc;
	TCF0.CTRLA=TC_CLKSEL_DIV1_gc;
}

int main(void)
{
	static uint8_t OldButtons;

	void InterpretCommand(char *cmdline)
	{
		uint8_t indeks;
		char *cmd=strtok(cmdline, " \r\n"); //Wydziel polecenie z przekazanej linii
		uint8_t max_indeks=sizeof(Polecenia)/sizeof(Polecenia[0]);
		for(indeks=0; indeks<max_indeks; indeks++)
		{
			if(strcmp_P(cmd, Polecenia[indeks].cmd)==0) //Przeszukaj list polece
			{
				Polecenia[indeks].Handler(cmdline);   //Wywoaj funkcj obsugi przesanego polecenia
				break;
			}
		}
		if(indeks == max_indeks)  //Jeli polecenie nieznane...
		{
			udi_cdc_write_buf(cmd, strlen(cmd));
			udi_cdc_write_buf(" - nieznane polecenie\r\n", 22); //Bd
		}
	}

	PMIC.CTRL = PMIC_LOLVLEN_bm | PMIC_MEDLVLEN_bm | PMIC_HILVLEN_bm; //Wczamy wszystkie poziomy przerwa (potrzebujemy tylko najniszy)
	sei();

	SPI_init();
	BatBackup_init();  //Zainicjuj RTC
	sysclk_init();
	udc_start();
	ADC_Init();
	Kbd_init();        //Zainicjuj przycisk Btn0 i Btn1

	while (1)
	{
		char Bufor[256];   //Bufor na odebranie linii
		uint8_t indeks=0;

		while(1)
		 {
			if(udi_cdc_is_rx_ready())
			{
				Bufor[indeks] = udi_cdc_getc();
				if(Bufor[indeks] == '\n')
				{
					if((indeks >= 1) && (Bufor[indeks-1] == '\r')) --indeks;
					Bufor[indeks]=0; //Wpisz kocowy znak NUL
					break;           //Odebrano pen lini
				}
				++indeks;
			}

			if(((Buttons ^ OldButtons) & 1) && (Buttons & 1))
			{
				if(TCC0.CTRLA == TC_CLKSEL_OFF_gc) TCC0.CTRLA=TC_CLKSEL_DIV256_gc;        //Taktowanie mniej wicej 62,5 kHz)
				 else
				 {
					 TCC0.CTRLA=TC_CLKSEL_OFF_gc;
					 DF_SynchBuffers();            //Natychmiast zapisuje stan bufora DataFLASH
				 }
			}

			if(((Buttons ^ OldButtons) & 2) && (Buttons & 2)) SendDataToADV();

			OldButtons=Buttons;
		 }
		InterpretCommand(Bufor);  //Zdekoduj przesane polecenie
	}
}

void main_suspend_action(void)
{
//Interfejs wchodzi w tryb powerdown
}

void main_resume_action(void)
{
//Wybudzenie interfejsu
}

void main_sof_action(void)
{
//	Odebrano ramk
}

bool main_cdc_enable(uint8_t port)
{
	// Otwarto port CDC
	return true;
}

void main_cdc_disable(uint8_t port)
{
	// Zamknito port CDC
}

void main_cdc_set_dtr(uint8_t port, bool b_enable)
{
	if (b_enable) {
		// Host otwar port COM
	}else{
		// Host zamkn port COM
	}
}

void uart_rx_notify(uint8_t port)
{
	//Opcjonalnie powiadomienie o nadejciu znaku
}

void uart_config(uint8_t port, usb_cdc_line_coding_t * cfg)
{
	//Konfiguracja w odpowiedzi na otwarcie CDC
}

void udi_cdc_write_txt(char *txt)
{
	udi_cdc_write_buf(txt, strlen(txt));
}

void CMD_date(char *param)
{
	struct calendar_date dt;                    //Struktura zawierajca dat i czas
	uint32_t timestamp=RTC32_get_counter();     //Pobierz czas z RTC
	calendar_timestamp_to_date(timestamp, &dt); //Zamie marker na format czytelny dla czowieka

	param=strtok(NULL, " :");
	if(param == NULL)
	{  //Odczytujemy aktualny czas
		char txtbuf[100];
		sprintf(txtbuf, "%02u:%02u:%02u %2u/%02u/%4u\r\n", dt.hour, dt.minute, dt.second, dt.date+1, dt.month+1, dt.year);      //Wywietl czas i dat
		udi_cdc_write_txt(txtbuf);
	} else
	{ //Ustawiamy czas w formacie hh:mm:ss [dd:mm:yy]
		dt.hour=atoi(param);
		param=strtok(NULL, ":");
		if(param)
		{
			dt.minute=atoi(param);
			param=strtok(NULL, ": ");
			if(param)
			{
				dt.second=atoi(param);
				param=strtok(NULL, ":/");
				if(param)
				{
					dt.date=atoi(param)-1;
					param=strtok(NULL, ":/");
					if(param)
					{
						dt.month=atoi(param)-1;
						param=strtok(NULL, ":/");
						if(param) dt.year=atoi(param);
						if(dt.year < 100) dt.year+=2000; //Jeli rok jest dwucyfrowy to mamy domylnie 20YY
					}
				}
			}
		}

		if(calendar_is_date_valid(&dt) == false)
			udi_cdc_write_txt("Podany czas jest nieprawidowy. Podaj czas w formacie hh:mm:ss [dd:mm:[yy]yy]\r\n");
			  else RTC32_set_counter(calendar_date_to_timestamp(&dt));
	}

}

void CMD_erase(char *param)
{
	uint8_t TimerClk=TCC0.CTRLA; //Zachowaj stan zegara timera
	TCC0.CTRLA=TC_CLKSEL_OFF_gc; //Wycz akwizycj danych
	udi_cdc_write_txt("Kasuje dane...");
	DataFLASH_SendCmdAddr(AT45DBX_CMDB_ER_CHIP, CS_High);  //Skasuj ca pami
	DataFLASH_WaitForBusy();                               //Zaczekaj na koniec operacji
	udi_cdc_write_txt("Dane skasowane.\r\n");
	DataFlash_Init();            //Znajd pierwsz woln pozycj do zapisu
	TCC0.CTRLA=TimerClk;         //Odtwrz zegar timera
}

void CMD_get(char *param)
{
	uint8_t Bufor[DataFLASH_PageSize];
	char txtbuf[100];
	__uint24 Adres=0;  //Adres w pamici DataFLASH
	DataLogger_t *dl;
	uint16_t Offset=0;
	_Bool koniec=false;

	uint8_t TimerClk=TCC0.CTRLA; //Zachowaj stan zegara timera
	TCC0.CTRLA=TC_CLKSEL_OFF_gc; //Wycz akwizycj danych na czas ich wysyania do PC
	DataFLASH_WaitForBusy();     //Zaczekaj na gotowo pamici

	udi_cdc_write_txt("Timestamp, CH0, CH1, CH2, CH3\r\n");
	do
	{
		DataFLASH_ReadSeq(Adres, Bufor, DataFLASH_PageSize); //Odczytaj stron pamici
		Adres+=DataFLASH_PageSize;
		while(Offset<DataFLASH_PageSize)
		{
			dl=(DataLogger_t*)&Bufor[Offset];
			Offset+=sizeof(DataLogger_t);
			if(dl->Timestamp == (uint32_t)-1)
			{
				koniec=true;   //Koniec danych
				break; //Zakocz ptl
			}
			sprintf(txtbuf, "%lu,%d,%d,%d,%d\r\n", dl->Timestamp, dl->CH[0], dl->CH[1], dl->CH[2], dl->CH[3]);
			udi_cdc_write_txt(txtbuf); //Wylij dane
		}
		Offset=0;
	} while(!koniec);

	TCC0.CTRLA=TimerClk;         //Odtwrz zegar timera
}

void CMD_start(char *param)
{
	TCC0.CTRLA=TC_CLKSEL_DIV256_gc;        //Taktowanie mniej wicej 62,5 kHz
	udi_cdc_write_txt("Akwizycja rozpoczeta\r\n");
}

void CMD_stop(char *param)
{
	TCC0.CTRLA=TC_CLKSEL_OFF_gc;  //Zatrzymaj timer
	DF_SynchBuffers();            //Natychmiast zapisuje stan bufora DataFLASH
	udi_cdc_write_txt("Akwizycja zakonczona\r\n");
}

void CMD_dump(char *param)
{
	uint8_t Bufor[16];
	char CharBuf[16*4+3];       //Bufor na budowany wiersz
	char chBuf[5];
	uint8_t asize;
	param=strtok(NULL, " ,;");
	if(param==NULL) return;      //Bd
	__uint24 start=atol(param);  //Pobierz adres startu
	param=strtok(NULL, " ,;");
	if(param==NULL) return;      //Bd
	__uint24 size=atol(param);   //Ile danych odczyta?

	uint8_t TimerClk=TCC0.CTRLA; //Zachowaj stan zegara timera
	TCC0.CTRLA=TC_CLKSEL_OFF_gc; //Wycz akwizycj danych na czas ich wysyania do PC
	DataFLASH_WaitForBusy();  //Zaczekaj na koniec operacji

	while(size)
	{
		if(size>16) asize=16; else asize=size;
		DataFLASH_ReadSeq(start, Bufor, asize); //Odczytaj dane z pamici DataFLASH

		CharBuf[0]=0;
		for(uint8_t i=0; i<asize; i++)   //Zamie odczytane dane na wiersz znakw ASCII
		{
			sprintf(chBuf, "%02X ", Bufor[i]);
			strcat(CharBuf,  chBuf);
		}
		strcat(CharBuf, "\r\n");         //Dodaj znak koca wiersza
		udi_cdc_write_buf(CharBuf, strlen(CharBuf)); //Wylij dane do komputera
		size-=asize;
		start+=asize;
	}

	TCC0.CTRLA=TimerClk;         //Odtwrz zegar timera
}

void CMD_Bufdump(char *param)
{
	char CharBuf[16*4+3];       //Bufor na budowany wiersz
	char chBuf[5];
	uint8_t asize, strona=0;
	param=strtok(NULL, " ,;");
	if(param != NULL) strona=atol(param);  //Pobierz adres strony

	TCC0.CTRLA=TC_CLKSEL_OFF_gc; //Wycz akwizycj danych na czas ich wysyania do PC
	DataFLASH_WaitForBusy();  //Zaczekaj na koniec operacji

	uint16_t size=DataFLASH_PageSize;
	if(strona == 0) DataFLASH_SendCmdAddr(AT45DBX_CMDC_RD_BUF1_AF_SM, 0, CS_Low); //Odczytujemy stron pamici
		else DataFLASH_SendCmdAddr(AT45DBX_CMDC_RD_BUF2_AF_SM, 0, CS_Low);

	while(size)
	{
		if(size>16) asize=16; else asize=size;

		CharBuf[0]=0;
		for(uint8_t i=0; i<asize; i++)   //Zamie odczytane dane na wiersz znakw ASCII
		{
			sprintf(chBuf, "%02X ", SPI_RW_Byte(0));
			strcat(CharBuf,  chBuf);
		}
		strcat(CharBuf, "\r\n");         //Dodaj znak koca wiersza
		udi_cdc_write_buf(CharBuf, strlen(CharBuf)); //Wylij dane do komputera
		size-=asize;
	}
}