/*
 * mp3_sinetest.c
 *
 * Created: 2013-10-28 20:52:52
 *  Author: tmf
 */


#include <avr/io.h>
#include <stdbool.h>
#include <util/delay.h>
#include "VS1003B.h"
#include "Plugin.h"

//Definicje pinw za pomoc ktrych podczony jest VS1003B
#define VS_DREQ    PIN3_bm
#define VS_xDCS    PIN2_bm
#define VS_xCS     PIN1_bm
#define VS_RESET   PIN4_bm
#define VS_TxD     PIN7_bm
#define VS_SCK     PIN5_bm
#define VS_RxD     PIN6_bm

#define VS_PORT    PORTC      //Port do ktrego jest podczony VS1003B
#define VS_USART   USARTC1    //USART przez ktry nastuje komunikacja

#define VS_SPI_CLK 1000000UL   //Taktowanie SPI dla VS1003B

#define DTMF_SPACE 57          //Czas odstpu pomidzy tonami w ms
#define DTMF_DURATION 72       //Czas trwania tonu w ms
#define DTMF_VOL   0xa0a0      //Gono tonw DTMF

const uint16_t __flash DTMFTones[][2]={{941, 1633}, {697, 1209}, {697, 1336}, {697, 1477}, {770, 1209}, {770, 1336}, {770, 1477},
	                                   {852, 1206}, {852, 1336}, {852, 1477}, {941, 1336}, {941, 1209}, {941, 1477}, {697, 1633},
									   {770, 1633}, {852, 1633}};

const uint8_t __flash DTMFCodes[]="D1234567890*#ABC";

void SPI_VS_init()
{
	VS_PORT.OUTSET=VS_xCS | VS_xDCS | VS_RESET;
	VS_PORT.DIRSET=VS_xDCS | VS_xCS | VS_RESET | VS_TxD | VS_SCK;   //Piny wyjciowe interfejsu
	VS_USART.CTRLC=USART_CMODE_MSPI_gc;             //Master SPI, MSB first, dane prbokwane na zboczu narastajcym

	#if F_CPU > VS_SPI_CLK
	VS_USART.BAUDCTRLA=F_CPU/VS_SPI_CLK/2 - 1;        //Oblicz taktowanie SPI
	#else
	VS_USART.BAUDCTRLA=0;
	#endif

	VS_USART.BAUDCTRLB=0;   //Taktowanie FCLK/2
	VS_USART.CTRLB=USART_RXEN_bm | USART_TXEN_bm;   //Odblokuj nadajnik i odbiornik
}

uint8_t SPI_VS_io(uint8_t data)
{
	VS_USART.DATA=data;
	while(!(VS_USART.STATUS & USART_TXCIF_bm));
	VS_USART.STATUS=USART_TXCIF_bm;    //Skasuj flag
	return VS_USART.DATA;
}

uint16_t SPI_VS_io_word(uint16_t word)
{
	uint16_t ret=SPI_VS_io(word >> 8) << 8;  //Wylij bardziej znaczcy bajt
	ret+=SPI_VS_io(word & 0xff);             //Wylij mniej znaczcy bajt
	return ret;
}

//Zresetuj koprocesor, zwr false, jeli ukad nie odpowiada
_Bool VS1003_Reset()
{
	VS_PORT.OUTCLR=VS_RESET;  //Reset ukadu
	_delay_ms(1);
	VS_PORT.OUTSET=VS_RESET;

	uint8_t del=255;
	while(((VS_PORT.IN & VS_DREQ) == 0) && (del))   //Czekamy na gotowo ukadu
	{
		_delay_us(100);
		del--;
	}
	return (del != 0);     //Zwr false jeli ukad nie odpowiedzia na RESET
}

//Dostp do interfejsu polece VS1003
static inline void VS1003_xCS_Enable(_Bool en)
{
	if(en)
	{
		VS_PORT.OUTSET=VS_xDCS;    //Deaktywuj xDCS
		VS_PORT.OUTCLR=VS_xCS;     //Aktywuj xCS
	} else VS_PORT.OUTSET=VS_xCS;
}

//Dostp do interfejsu danych VS1003
static inline void VS1003_xDCS_Enable(_Bool en)
{
	if(en)
	{
		VS_PORT.OUTSET=VS_xCS;    //Deaktywuj xCS
		VS_PORT.OUTCLR=VS_xDCS;   //Aktywuj xDCS
	} else VS_PORT.OUTSET=VS_xDCS;
}

static inline void VS_1003_DREQ_High()
{
	while((VS_PORT.IN & VS_DREQ) == 0);
}

void VS1003_WriteReg(uint8_t cmd, uint16_t value)
{
	VS1003_xCS_Enable(true);       //Aktywuj DCI
	SPI_VS_io(VS_WRITE_CMD);
	SPI_VS_io(cmd);
	SPI_VS_io_word(value);         //Domylna konfiguracja
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();           //Zaczekaj na koniec realizacji polecenia
}

uint16_t VS1003_ReadReg(uint8_t cmd)
{
	VS1003_xCS_Enable(true);       //Aktywuj DCI
	SPI_VS_io(VS_READ_CMD);
	SPI_VS_io(cmd);
	uint16_t ret=SPI_VS_io_word(0);
	VS1003_xCS_Enable(false);
	VS_1003_DREQ_High();           //Zaczekaj na koniec realizacji polecenia
	return ret;
}

_Bool VS1003_init()
{
	SPI_VS_init();                            //Inicjalizacja interfejsu
	if(VS1003_Reset() == false) return false; //Zresetuj ukad
	return true;
}

void VS_LoadPlugin()
{
	int i;
	for (i=0;i<CODE_SIZE;i++)
	{
		VS1003_WriteReg(atab[i], dtab[i]);
	}
}

void VS_GenDTMF(uint8_t code)
{
	VS1003_WriteReg(VS_AICTRL0, DTMFTones[code][0]);
	VS1003_WriteReg(VS_AICTRL1, DTMFTones[code][1]);
	VS1003_WriteReg(VS_AICTRL2, DTMF_VOL);              //Wcz ton
	_delay_ms(DTMF_DURATION);   //Czas trwania tonu
	VS1003_WriteReg(VS_AICTRL2, 0x0000);              //Cisza
	_delay_ms(DTMF_SPACE);    //Odstp pomidzy znakami
}

int8_t DTMFToIndex(uint8_t DTMF)
{
	uint8_t index=0;
	while(index<=15)
	{
		if(DTMFCodes[index] == DTMF) return index;
		index++;
	}
	return -1; //Bd
}

void VS_SendDTMF(char *DTMF)
{
	uint8_t code;
	while(*DTMF)
	{
		code=DTMFToIndex(*DTMF);  //Zamie kod DTMF na indeks do tabeli tonw
		VS_GenDTMF(code);
		DTMF++; //Kolejny znak
	}
}

char dial[]="D01234567890AB12345678";

int main(void)
{
	VS1003_init();
	VS_LoadPlugin();

	VS1003_WriteReg(VS_AICTRL3, 0xffff);
	VS1003_WriteReg(VS_AIADDR, 0x0030);

	VS_SendDTMF(dial);

    while(1)
    {
		VS_SendDTMF(dial);
		_delay_ms(2000);
    }
}