/*
 * RC5.c
 *
 * Created: 2013-02-17 14:03:51
 *  Author: tmf
 */ 


#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>

#include "usart.h"
#include "RingBuffer.h"

#define IR_HALF_BIT      889     //Czas trwania pbitu
#define IR_BIT_TOLERANCE 100     //Tolerancja czasu trwania pbitu
#define IR_BITSNO         13     //Liczba bitw kodu - 9 + 1 prebit

volatile uint16_t IR_RecCmd;     //Odebrana komenda z pilota
volatile uint8_t  IR_recpos;     //Nr aktualnie odbieranego bitu

CircBuffer IR_CMD_Buffer; //Instancja bufora koowego przechowujcego polecenia

static inline uint16_t IR_CalcTime(uint16_t time)
{
	return time*(F_CPU/1000000UL)/4;   //Przelicz czas w mikrosekundach na tyknicia timera
	                                   //Przy timerze taktowanym CLKPER/4
}

static uint8_t IR_Polarity;           //Stan dekodera Manchester

ISR(PORTD_INT0_vect)
{
	TCD0.CNT=0;
	if(IR_recpos==0)           //Inicjujemy odbir
	{
		IR_RecCmd=1;
	    IR_Polarity=1;
	    TCD0.INTFLAGS=TC0_CCAIF_bm;        //Skasuj ewentualn flag przerwania CCA
		TCD0.INTFLAGS=TC0_OVFIF_bm;        //i nadmiaru
	    TCD0.INTCTRLB=TC_CCAINTLVL_LO_gc;  //Wcz przerwania CCA
		TCD0.INTCTRLA=TC_OVFINTLVL_LO_gc;  //Wcz przerwanie nadmiaru
	}				
}

ISR(TCD0_OVF_vect)
{
	PORTD_PIN2CTRL=(PORTD_PIN2CTRL & ~PORT_ISC_gm) | PORT_ISC_FALLING_gc; //Wcz zdarzenia na zboczu opadajcym
	IR_recpos=0;
	TCD0.INTCTRLB=TC_CCAINTLVL_OFF_gc;   //Wycz przerwanie CCA
	TCD0.INTCTRLA=TC_OVFINTLVL_OFF_gc;   //Wycz przerwanie nadmiaru
}

ISR(TCD0_CCA_vect)
{
	uint8_t tmp=PORTD_IN;
	if(tmp & PIN2_bm)
	{
		IR_Polarity=1;
		PORTD_PIN2CTRL=(PORTD_PIN2CTRL & ~PORT_ISC_gm) | PORT_ISC_FALLING_gc; //Przerwania generuje zbocze opadajce
	} else
	{
		IR_Polarity=0;
		PORTD_PIN2CTRL=(PORTD_PIN2CTRL & ~PORT_ISC_gm) | PORT_ISC_RISING_gc; //Przerwania generuje zbocze narastajce
	}
	
	IR_RecCmd<<=1;
    IR_RecCmd|=IR_Polarity;   //Wsu warto bitu
	++IR_recpos;
	if(IR_recpos==IR_BITSNO)
	{
		PORTD_PIN2CTRL=(PORTD_PIN2CTRL & ~PORT_ISC_gm) | PORT_ISC_FALLING_gc; //Wcz zdarzenia na zboczu opadajcym
		TCD0.INTCTRLB=TC_CCAINTLVL_OFF_gc;   //Wyczamy przerwanie CCA
		TCD0.INTCTRLA=TC_OVFINTLVL_OFF_gc;   //Wycz przerwanie nadmiaru
		IR_recpos=0;
		cb_Add(&IR_CMD_Buffer, IR_RecCmd);     //Dodaj odczytane polecenie do kolejki
	}
}

void IR_init()
{
	PORTD.PIN2CTRL=PORT_OPC_PULLUP_gc | PORT_ISC_FALLING_gc;  //Pulup na pinie odbiornika IR, zbocze opadajce wywouje zdarzenie
	PORTD.INT0MASK=PIN2_bm;                 //Wcz przerwania pinu
	PORTD.INTCTRL=PORT_INT0LVL_LO_gc;
	TCD0.CTRLB=TC_WGMODE_NORMAL_gc;         //Tryb pracy normalny
	TCD0.PER=IR_CalcTime(3*IR_HALF_BIT);    //Okres timera
	TCD0.CCA=IR_CalcTime(1.5*IR_HALF_BIT);  //Moment prbkowania - 3/4 bitu 
	TCD0.CTRLA=TC_CLKSEL_DIV4_gc;
	PMIC_CTRL=PMIC_LOLVLEN_bm;             //Odblokuj przerwania niskiego poziomu
}

void usart_init()
{
	USARTC0.CTRLB=USART_TXEN_bm;         //Wcz nadajnik USART
	USARTC0.CTRLC=USART_CHSIZE_8BIT_gc;  //Ramka 8 bitw, bez parzystoci, 1 bit stopu
	usart_set_baudrate(&USARTC0, 9600, F_CPU);
	PORTC_DIRSET=PIN3_bm;                //Pin TxD musi by wyjciem
}

int main(void)
{
	char bufor[13];
	
	IR_init();
	usart_init();
	sei();
	
	while(1)
	{
		if(cb_IsEmpty(&IR_CMD_Buffer)==false)
		{
			CB_Element cmd=cb_Read(&IR_CMD_Buffer);
			ultoa(cmd, bufor, 16);
			USART_send(&USARTC0, bufor);
			USART_putchar(&USARTC0, '\n');
			USART_putchar(&USARTC0, '\r');
		}
	}
}
