/*
 * IRrecorder.c
 *
 * Created: 2013-02-18 21:09:13
 *  Author: tmf
 */ 


#include <avr/io.h>
#include <string.h>
#include <avr/interrupt.h>
#include <util/delay.h>

#define F_CARRIER  38000UL       //Nona dla IR
#define IR_BUFFER_MAX     200    //Maksymalna liczba prbek w buforze
#define IR_MAX_CAPT       10     //Liczba przechwyconych sekwencji

uint16_t IR_Buffer[IR_BUFFER_MAX]; //Bufor na prbki

typedef struct  
{
	uint8_t  len;                   //Liczba przechwyconych bajtw
	uint16_t vals[IR_BUFFER_MAX];   //Markery czasowe
} IR_Capt_t;

IR_Capt_t Capt_Keys[IR_MAX_CAPT];  //Przechwycone sekwencje
volatile uint8_t Capt_seq;

enum IR_CAPT_STATUS {ST_OFF, ST_Started};

enum IR_CAPT_STATUS IR_Status;  //Stan przechwytywania IR
uint8_t OutState;   //Stan modulacji

ISR(PORTC_INT0_vect)
{
	if(IR_Status==ST_OFF)
	{
		TCD0.CNT=0;              //Zainicjuj timer
		TCD0.CTRLA=TC_CLKSEL_DIV8_gc;
	}
	IR_Status=ST_Started;
}

ISR(TCD0_OVF_vect)
{ //Przerwanie wywoywane, kiedy skoczy si czas przechwytywania danych lub okresowo
	TCD0.CTRLA=TC_CLKSEL_OFF_gc;
	DMA.CH0.CTRLA=DMA.CH0.CTRLA & ~DMA_CH_ENABLE_bm; //Zatrzymaj DMA
	IR_Status=ST_OFF;
	Capt_Keys[Capt_seq].len=IR_BUFFER_MAX-DMA.CH0.TRFCNT;  //Liczba prbek
	memcpy(Capt_Keys[Capt_seq].vals, IR_Buffer, Capt_Keys[Capt_seq].len); //Skopiuj prbki
	++Capt_seq;
}


ISR(TCD0_CCA_vect)
{
	if(OutState) TCC0.CTRLB=TC_WGMODE_SS_gc | TC0_CCDEN_bm; //Wcz modulacj - nadajemy 1
	 else TCC0.CTRLB=TC_WGMODE_SS_gc; //Wycz modulacj - nadajemy 0
	OutState^=1;
}

void IR_Rec_init()
{
	PORTC.PIN2CTRL=PORT_OPC_PULLUP_gc | PORT_ISC_BOTHEDGES_gc;  //Pulup na pinie odbiornika IR, oba zbocza wywouje zdarzenie
	PORTC.INT0MASK=PIN2_bm;            //Wcz przerwania
	PORTC.INTCTRL=PORT_INT0LVL_LO_gc;
	EVSYS.CH0MUX=EVSYS_CHMUX_PORTC_PIN2_gc;  //Pocz pin z CH0
	TCD0.CTRLB=TC_WGMODE_NORMAL_gc | TC0_CCAEN_bm;    //Tryb pracy normalny, odblokuj kana CCA
	TCD0.CTRLD=TC_EVACT_CAPT_gc | TC_EVSEL_CH0_gc;    //Tryb capture z kanau 0
	TCD0.INTCTRLA=TC_OVFINTLVL_LO_gc;
    TCD0.INTCTRLB=0;                       //Wycz wszystkie przerwania CC
	DMA.CTRL=DMA_ENABLE_bm;
	DMA.CH0.SRCADDR0=(uint8_t)(uint16_t)&TCD0_CCA;         //Adres rda danych
	DMA.CH0.SRCADDR1=(uint8_t)((uint16_t)&TCD0_CCA >>8);
	DMA.CH0.SRCADDR2=0;
	DMA.CH0.DESTADDR0=(uint8_t)(uint16_t)IR_Buffer;        //Adres przeznaczenia
	DMA.CH0.DESTADDR1=(uint8_t)((uint16_t)IR_Buffer >>8);
	DMA.CH0.DESTADDR2=0;
	DMA.CH0.TRFCNT=IR_BUFFER_MAX;
	DMA.CH0.REPCNT=0;
	DMA.CH0.ADDRCTRL=DMA_CH_SRCDIR_INC_gc | DMA_CH_SRCRELOAD_BURST_gc | DMA_CH_DESTDIR_INC_gc | DMA_CH_DESTRELOAD_BLOCK_gc;
	DMA.CH0.TRIGSRC=DMA_CH_TRIGSRC_TCD0_CCA_gc;  //Transfer wyzwoli zdarzenie przechwycenia czego przez TCD0 kana CCA
	DMA.CH0.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_2BYTE_gc; 
}

void IR_Play(uint8_t sampleno)
{
	OutState=1;
	TCC0.PER=F_CPU/F_CARRIER-1;
	TCC0.CCD=TCC0.PER/2;           //Wypenienie 50%
	TCC0.CTRLA=TC_CLKSEL_DIV1_gc;  //Wcz timer
	PORTC.INT0MASK=0;              //Wycz przerwania pinu
	PORTC.OUTCLR=PIN3_bm;
	PORTC.DIRSET=PIN3_bm;          //PC3 jest wyjciem w stanie 0
	TCD0.CTRLB=TC_WGMODE_NORMAL_gc | TC0_CCAEN_bm;             //Tryb pracy normalny, odblokuj kana CCA - pin 0
	TCD0.INTCTRLA=TC_OVFINTLVL_OFF_gc | TC_ERRINTLVL_OFF_gc;   //Wycz przerwanie OVF i ERR
	TCD0.INTCTRLB=TC_CCAINTLVL_LO_gc;
	TCD0.CCA=0;
	TCD0.CTRLD=TC_EVACT_OFF_gc | TC_EVSEL_OFF_gc; //Wycz inne funkcje timera
	
	DMA.CTRL=DMA_ENABLE_bm;
	DMA.CH0.DESTADDR0=(uint8_t)(uint16_t)&TCD0_CCA;         //Adres rda danych
	DMA.CH0.DESTADDR1=(uint8_t)((uint16_t)&TCD0_CCA >>8);
	DMA.CH0.DESTADDR2=0;
	DMA.CH0.SRCADDR0=(uint8_t)(uint16_t)Capt_Keys[sampleno].vals;  //Adres skd bierzemy prbki
	DMA.CH0.SRCADDR1=(uint8_t)((uint16_t)Capt_Keys[sampleno].vals >> 8);
	DMA.CH0.SRCADDR2=0;
	DMA.CH0.TRFCNT=Capt_Keys[sampleno].len;  //Liczba prbek
	DMA.CH0.REPCNT=0;
	DMA.CH0.ADDRCTRL=DMA_CH_SRCDIR_INC_gc | DMA_CH_SRCRELOAD_BLOCK_gc | DMA_CH_DESTDIR_INC_gc | DMA_CH_DESTRELOAD_BURST_gc;
	DMA.CH0.TRIGSRC=DMA_CH_TRIGSRC_TCD0_CCA_gc; //Transfer wyzwoli zdarzenie przepenienia
	DMA.CH0.CTRLA=DMA_CH_ENABLE_bm | DMA_CH_SINGLE_bm | DMA_CH_BURSTLEN_2BYTE_gc | DMA_CH_TRFREQ_bm; //Pierwsze zdarzenie wyzwol rcznie
	TCD0.CTRLA=TC_CLKSEL_DIV8_gc; //Rozpocznij nadawanie
	while(!(DMA.CH0.CTRLB & DMA_CH_TRNIF_bm)); //Zaczekaj na koniec
	TCC0.CTRLB=TC_WGMODE_SS_gc;    //Wycz generowanie nonej
	TCD0.CTRLA=TC_CLKSEL_OFF_gc;   //Wycz timer
	DMA.CH0.CTRLB=DMA_CH_TRNIF_bm; //Skasuj flag
}

void io_init()
{
	PORTR_DIRSET=PIN0_bm;              //Pin z LED na wyjcie
	PORTE_PIN5CTRL=PORT_OPC_PULLUP_gc; //Podciganie dla klawisza SW0
}

int main(void)
{
	io_init();
	IR_Rec_init();
	PMIC_CTRL=PMIC_LOLVLEN_bm;             //Odblokuj przerwania niskiego poziomu
	sei();
	while(Capt_seq==0)
	{
		PORTR_OUTTGL=PIN0_bm; //zmie stan LED
		_delay_ms(500);
	}
	PORTR_OUTCLR=PIN0_bm;
    while(1)
    {
        if((PORTE_IN & PIN5_bm)==0)
		{
			IR_Play(0);
			_delay_ms(100);
		}			
    }
}