// LISTING 10.27. Klasa MyRestEndpoint zwracająca dane zgłoszeń w formacie JSON

package pl.helion.rest;

import com.atlassian.jira.bc.issue.search.SearchService;
import com.atlassian.jira.issue.search.SearchResults;
import com.atlassian.jira.web.bean.PagerFilter;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.search.SearchException;
import com.atlassian.jira.user.ApplicationUser;
import pl.helion.rest.model.IssueItem;
import pl.helion.rest.model.Issues;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;

/**
 * Klasa REST Endpoint z metodą wystawiającą dane o zgłoszeniach.
 */
@Path("/")
public class MyRestEndpoint {

    /**
     * Metoda wystawia dane o zgłoszeniach w formacie JSON.
     * @param created — data w formacie yyyy-MM-dd umieszczana jako atrybut w adresie URL.
     * @return — dane o zgłoszeniach w formacie JSON według deklaracji w @Produces.
     * @throws SearchException
     */
    @GET
    @Path("/getissue/{created}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getIssueByCreationDate(@PathParam("created") String created) throws SearchException {

        // 1. Parsowanie daty pobranej z atrybutu URL do obiektu LocalDate.
        LocalDate localDate = LocalDate.parse(created);
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");

        // 2. Pobranie obiektu typu interfejsowego ApplicationUser bieżącego użytkownika.
        ApplicationUser currentUser = ComponentAccessor.getJiraAuthenticationContext().getLoggedInUser();

        // 3. Treść zapytania JQL, które zwróci zgłoszenia z danego dnia według schematu: od [data] [00:00] do [data + 1 dzień] [00:00]
        String jqlQuery = "created >= " + localDate.format(formatter) + " AND created <= " + localDate.plusDays(1).format(formatter);

        // 4. Pobranie komponentu SearchService.
        SearchService searchService = ComponentAccessor.getComponentOfType(SearchService.class);

        // 5. Parsowanie zapytania JQL.
        SearchService.ParseResult parseResult = searchService.parseQuery(currentUser, jqlQuery);

        // 6. Uruchomienie zapytania JQL.
        SearchResults results = searchService.search(currentUser, parseResult.getQuery(), PagerFilter.getUnlimitedFilter());

        // 7. Zwrócenie rezultatu w postaci listy obiektów typu interfejsowego Issue.
        List<Issue> issueResult = results.getIssues();

        // 8. Inicjalizacja obiektu Issues do przechowywania obiektów zgłoszeń typu IssueItem.
        Issues issues = new Issues();

        // 9. Przegląd w pętli wszystkich znalezionych w JQL zgłoszeń, utworzenie dla nich obiektów IssueItem i dodanie do listy w obiekcie Issues.
        issueResult.forEach(
                issue -> {
                    issues.setIssue(new IssueItem(issue.getId(), issue.getKey(), issue.getSummary(), issue.getCreated()));
                });

        // 10. Zwrócenie wyniku.
        return Response.ok(issues).build();
    }
}