﻿using System;
using System.Collections.Generic;
using System.Linq;
using PortalUslug.Models;

namespace PortalUslug.Repositories
{
    /// <summary>
    /// Klasa repozytorium usługi.
    /// </summary>
    public class ServiceRepository : IServiceRepository
    {   
        /// <summary>
        /// Obiekt klasy kontekstowej.
        /// </summary>
        private PortalUslugContext _db;

        /// <summary>
        /// Konstruktor repozytorium usług.
        /// </summary>
        public ServiceRepository()
        {
            _db = new PortalUslugContext();
        }

        /// <summary>
        /// Pobranie usługi o podanym identyfikatorze.
        /// </summary>
        /// <param name="id">Identyfikator usługi.</param>
        /// <returns>Usługa o podanym identyfikatorze.</returns>
        public Service GetServiceById(int id)
        {
            return _db.Services.FirstOrDefault(service => service.Id == id);
        }

        /// <summary>
        /// Sprawdzenie, czy użytkownik o podanym identyfikatorze dodał usługę/-i.
        /// </summary>
        /// <param name="userId">Identyfikator użytkownika.</param>
        /// <returns>True, jeśli użytkownik dodał usługę/-i.</returns>
        public bool HasUserServices(int userId)
        {
            return _db.Services.Where(u => u.UserId == userId).Count() > 0;
        }

        /// <summary>
        /// Dodanie usługi.
        /// </summary>
        /// <param name="service">Dodawana usługa.</param>
        public void Add(Service service)
        {
            _db.Services.Add(service);
        }

        /// <summary>
        /// Usunięcie usługi.
        /// </summary>
        /// <param name="service">Usuwana usługa.</param>
        public void Delete(Service service)
        {
            _db.Services.Remove(service);
        }

        /// <summary>
        /// Zapisanie zmian.
        /// </summary>
        public void SaveChanges()
        {
            _db.SaveChanges();
        }

        /// <summary>
        /// Pobranie aktywnych usług.
        /// </summary>
        /// <returns>Aktywne usługi.</returns>
        public IQueryable<ServiceViewModel> GetActiveService()
        {
            var services = from p in _db.Services.Where(u => u.ExpirationDate > DateTime.Now).OrderByDescending(o => o.PostedDate)
                           select new ServiceViewModel
                           {
                               IPAddress = p.IPAddress,
                               PostedDate = p.PostedDate,
                               ExpirationDate = p.ExpirationDate,
                               Id = p.Id,
                               CategoryId = p.CategoryId,
                               UserId = p.UserId,
                               Content = p.Content,
                               Name = p.Name,
                               CategoryName = p.Categories.Name,
                               ServiceProvider = _db.ServiceProviders.FirstOrDefault(u => u.UserId == p.UserId).Name,
                               IsActive = p.ExpirationDate > DateTime.Now ? "Tak" : "Nie"
                           };

            return services;
        }

        /// <summary>
        /// Pobranie wszystkich usług.
        /// </summary>
        /// <returns>Wszystkie usługi.</returns>
        public IQueryable<ServiceViewModel> GetAllService()
        {
            var services = from p in _db.Services.OrderByDescending(o => o.PostedDate)
                           select new ServiceViewModel
                           {
                               IPAddress = p.IPAddress,
                               PostedDate = p.PostedDate,
                               ExpirationDate = p.ExpirationDate,
                               Id = p.Id,
                               CategoryId = p.CategoryId,
                               UserId = p.UserId,
                               Content = p.Content,
                               Name = p.Name,
                               CategoryName = p.Categories.Name,
                               ServiceProvider = _db.ServiceProviders.FirstOrDefault(u => u.UserId == p.UserId).Name,
                               IsActive = p.ExpirationDate > DateTime.Now ? "Tak" : "Nie"
                           };

            return services;
        }

        /// <summary>
        /// Pobranie usługi o podanym identyfikatorze.
        /// </summary>
        /// <param name="id">Identyfikator usługi.</param>
        /// <returns>Usługa o podanym identyfikatorze.</returns>
        public ServiceViewModel GetServiceViewModelById(int id)
        {
            var service = from p in _db.Services
                          where p.Id == id
                          select new ServiceViewModel
                          {
                              Id = p.Id,
                              PostedDate = p.PostedDate,
                              ExpirationDate = p.ExpirationDate,
                              CategoryId = p.CategoryId,
                              UserId = p.UserId,
                              Content = p.Content,
                              Name = p.Name,
                              CategoryName = p.Categories.Name,
                              ServiceProvider = _db.ServiceProviders.FirstOrDefault(u => u.UserId == p.UserId).Name,
                              IsActive = p.ExpirationDate > DateTime.Now ? "Tak" : "Nie"
                          };

            return service.FirstOrDefault();
        }


        /// <summary>
        /// Pobieranie aktywnych usług usługodawcy o podanym identyfikatorze użytkownika.
        /// </summary>
        /// <param name="userId">Identyfikator użytkownika.</param>
        /// <returns>Aktywne usługi użytkownika o podanym identyfikatorze użytkownika.</returns>
        public IQueryable<ServiceViewModel> GetActiveServicesByUserId(int userId)
        {
            var services = from p in _db.Services
                           where p.UserId == userId && p.ExpirationDate > DateTime.Now
                           select new ServiceViewModel
                           {
                               IPAddress = p.IPAddress,
                               PostedDate = p.PostedDate,
                               ExpirationDate = p.ExpirationDate,
                               Id = p.Id,
                               CategoryId = p.CategoryId,
                               UserId = p.UserId,
                               Content = p.Content,
                               Name = p.Name,
                               CategoryName = p.Categories.Name,
                               ServiceProvider = _db.ServiceProviders.FirstOrDefault(u => u.UserId == p.UserId).Name,
                               IsActive = p.ExpirationDate > DateTime.Now ? "Tak" : "Nie"
                           };
            return services;
        }

        /// <summary>
        /// Pobranie nowych usług.
        /// </summary>
        /// <returns>Pięć nowych usług.</returns>
        public IQueryable<ServiceViewModel> GetNewServices()
        {
            var services = from p in _db.Services
                           where p.ExpirationDate > DateTime.Now
                           orderby p.PostedDate descending
                           select new ServiceViewModel
                           {
                               PostedDate = p.PostedDate,
                               Id = p.Id,
                               UserId = p.UserId,
                               Name = p.Name,
                               ServiceProvider = _db.ServiceProviders.FirstOrDefault(u => u.UserId == p.UserId).Name
                           };

            return services.Take(5);
        }
    }
}