﻿using PortalUslug.Repositories;
using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Web.Mvc;
using System.Linq;
using MvcContrib.Pagination;
using MvcContrib.UI.Grid;

namespace PortalUslug.Models
{
    [MetadataType(typeof(ServiceMetaData))]
    public class Service
    {
        public Service()
        {
            this.Comments = new HashSet<Comment>();
        }

        public int Id { get; set; }
        public string Name { get; set; }
        public string Content { get; set; }
        public DateTime PostedDate { get; set; }
        public DateTime ExpirationDate { get; set; }
        public string IPAddress { get; set; }
        public int UserId { get; set; }
        public int CategoryId { get; set; }

        public virtual Category Categories { get; set; }
        public virtual ICollection<Comment> Comments { get; set; }
    }

    public class ServiceMetaData
    {
        [Required(ErrorMessage = "Nazwa jest wymagana.")]
        [Display(Name = "Nazwa")]
        public string Name { get; set; }

        [Required(ErrorMessage = "Treść jest wymagana.")]
        [Display(Name = "Treść usługi")]
        public string Content { get; set; }

        [ScaffoldColumn(false)]
        [Display(Name = "Kategoria")]
        public int CategoryId { get; set; }

        [DataType(DataType.DateTime)]
        [Display(Name = "Data umieszczenia")]
        public DateTime PostedDate { get; set; }

        [DataType(DataType.DateTime)]
        [Display(Name = "Data wygaśnięcia")]
        public DateTime ExpirationDate { get; set; }
    }

    public class AddServiceModel
    {
        public Service Service { get; set; }

        [ScaffoldColumn(false)]
        [Display(Name = "Liczba dni")]
        [RegularExpression(@"^[1-9]\d*$", ErrorMessage = "To nie jest liczba całkowita.")]
        public int? DaysNumber { get; set; }
    }

    public class ServiceViewModel
    {
        [ScaffoldColumn(false)]
        public int Id { get; set; }

        [DataType(DataType.DateTime)]
        [Display(Name = "Data umieszczenia")]
        public DateTime PostedDate { get; set; }

        [DataType(DataType.DateTime)]
        [Display(Name = "Data wygaśnięcia")]
        public DateTime ExpirationDate { get; set; }

        [Display(Name = "Nazwa")]
        public string Name { get; set; }

        [Display(Name = "Treść usługi")]
        public string Content { get; set; }

        [ScaffoldColumn(false)]
        public int CategoryId { get; set; }

        [Display(Name = "Kategoria")]
        public string CategoryName { get; set; }

        [ScaffoldColumn(false)]
        public int UserId { get; set; }

        [Display(Name = "Usługodawca")]
        public string ServiceProvider { get; set; }

        [Display(Name = "Adres IP")]
        public string IPAddress { get; set; }

        [ScaffoldColumn(false)]
        public string IsActive { get; set; }
    }

    public class ServiceFilterViewModel
    {
        private int selectedCategoryId = -1;
        private int selectedProviderId = -1;

        /// <summary>
        /// Lista kategorii usług.
        /// </summary>
        public List<SelectListItem> Categories { get; set; }

        /// <summary>
        /// Lista usługodawców.
        /// </summary>
        public List<SelectListItem> ServiceProviders { get; set; }

        /// <summary>
        /// Nazwa usługi.
        /// </summary>
        public string ServiceName { get; set; }

        /// <summary>
        /// Treść usługi.
        /// </summary>
        public string ServiceContent { get; set; }

        /// <summary>
        /// ID wybranej kategorii usługi.
        /// </summary>
        public int SelectedCategoryId
        {
            get
            {
                return selectedCategoryId;
            }
            set
            {
                selectedCategoryId = value;
            }
        }

        /// <summary>
        /// ID wybranego usługodawcy.
        /// </summary>
        public int SelectedProviderId
        {
            get
            {
                return selectedProviderId;
            }
            set
            {
                selectedProviderId = value;
            }
        }

        /// <summary>
        /// Funkcja przygotowująca dane wypełniające listy rozwijane.
        /// </summary>
        public void Fill()
        {
            // Repozytorium kategorii usług.
            CategoryRepository _categoryRepo = new CategoryRepository();

            // Repozytorium usługodawców.
            ServiceProviderRepository _providerRepo = new ServiceProviderRepository();

            // Wypełnienie listy kategorii usług danymi w postaci: nazwa kategorii - ID kategorii
            Categories = _categoryRepo.GetAllCategories()
                            .Select(a =>
                                new
                                {
                                    a.Name,
                                    a.Id
                                }
                            )
                            .ToList()
                            .Select(a =>
                                new SelectListItem
                                {
                                    Text = a.Name,
                                    Value = a.Id.ToString(),
                                    Selected = a.Id == selectedCategoryId
                                }).ToList();

            // Wypełnienie listy usługodawców danymi w postaci: nazwa usługodawcy - ID usługodawcy
            ServiceProviders = _providerRepo.GetServiceProviders()
                            .Select(a =>
                                new
                                {
                                    a.Name,
                                    a.UserId
                                }
                            )
                            .ToList()
                            .Select(a =>
                               new SelectListItem
                               {
                                   Text = a.Name,
                                   Value = Convert.ToString(a.UserId),
                                   Selected = a.UserId == SelectedProviderId
                               }).ToList();
        }
    }

    public class ServiceListContainerViewModel
    {
        public IPagination<ServiceViewModel> ServicePagedList { get; set; }
        public ServiceFilterViewModel ServiceFilterViewModel { get; set; }
        public GridSortOptions GridSortOptions { get; set; }
    }



}
