﻿using PortalUslug.Models;
using System.Linq;
using System.Web.Security;
using WebMatrix.WebData;

namespace PortalUslug.Repositories
{
    /// <summary>
    /// Klasa repozytorium usługobiorcy.
    /// </summary>
    public class CustomerRepository : ICustomerRepository
    {
        /// <summary>
        /// Obiekt klasy kontekstowej.
        /// </summary>
        private PortalUslugContext _db;

        /// <summary>
        /// Konstruktor repozytorium usługobiorcy.
        /// </summary>
        public CustomerRepository()
        {
            _db = new PortalUslugContext();
        }

        /// <summary>
        /// Pobranie usługobiorcy o podanym identyfikatorze.
        /// </summary>
        /// <param name="id">Identyfikator usługobiorcy.</param>
        /// <returns>Usługobiorca o podanym identyfikatorze.</returns>
        public Customer GetCustomerById(int id)
        {
            return _db.Customers.Find(id);
        }

        /// <summary>
        /// Pobranie usługobiorcy o podanym identyfikatorze użytkownika.
        /// </summary>
        /// <param name="id">Identyfikator użytkownika.</param>
        /// <returns> Usługobiorca o podanym identyfikatorze użytkownika.</returns>
        public Customer GetCustomerByUserId(int id)
        {
            return _db.Customers.FirstOrDefault(customer => customer.UserId == id);
        }

        /// <summary>
        /// Pobranie usługobiorców z newsletterem.
        /// </summary>
        /// <returns>Usługobiorcy z newsletterem.</returns>
        public IQueryable<Customer> GetAllCustomersWithNewsletter()
        {
            return _db.Customers.Where(u => u.Newsletter && u.IsConfirmed);
        }

        /// <summary>
        /// Sprawdzenie, czy usługobiorca o podanym identyfikatorze użytkownika jest usługobiorcą.
        /// </summary>
        /// <param name="id">Identyfikator użytkownika.</param>
        /// <returns>True, jeśli użytkownik o podanym identyfikatorze jest usługobiorcą.</returns>
        public bool IsCustomer(int id)
        {
            return _db.Customers.FirstOrDefault(u => u.UserId == id) != null ? true : false;
        }

        /// <summary>
        /// Sprawdzenie, czy usługobiorca o podanym identyfikatorze użytkownika ma potwierdzone konto.
        /// </summary>
        /// <param name="id">Identyfikator użytkownika.</param>
        /// <returns>True, jeśli użytkownik o podanym identyfikatorze ma potwierdzone konto.</returns>
        public bool IsConfirmed(int id)
        {
            return _db.Customers.FirstOrDefault(u => u.UserId == id).IsConfirmed;
        }

        /// <summary>
        /// Dodanie usługobiorcy.
        /// </summary>
        /// <param name="customer">Dodawany usługobiorca.</param>
        public void Add(Customer customer)
        {
            _db.Customers.Add(customer);
        }

        /// <summary>
        /// Usunięcie usługobiorcy.
        /// </summary>
        /// <param name="customer">Usuwany usługobiorca.</param>
        public void Delete(Customer customer)
        {
             string name = ((SimpleMembershipProvider)Membership.Provider).GetUserNameFromId(customer.UserId);
            _db.Customers.Remove(customer);
            _db.SaveChanges();

            Roles.RemoveUserFromRole(name, "Customer");
            ((SimpleMembershipProvider)Membership.Provider).DeleteAccount(name);
            ((SimpleMembershipProvider)Membership.Provider).DeleteUser(name, true);
        }

        /// <summary>
        /// Zapisanie zmian.
        /// </summary>
        public void SaveChanges()
        {
            _db.SaveChanges();
        }

        /// <summary>
        /// Pobranie wszystkich usługobiorców.
        /// </summary>
        /// <returns>Usługobiorcy.</returns>
        public IQueryable<CustomerViewModel> GetAllCustomers()
        {
            var customers = from p in _db.Customers
                            select new CustomerViewModel
                            {
                                Id = p.Id,
                                FirstName = p.FirstName,
                                LastName = p.LastName,
                                City = p.City,
                                Street = p.Street,
                                ZipCode = p.ZipCode,
                                UserId = p.UserId,
                                IsActive = p.IsConfirmed ? "Tak" : "Nie",
                                IsConfirmed = p.IsConfirmed,
                                RegistrationDate = p.RegistrationDate
                            };

            return customers;
        }
    }
}