//
//  Xboxhidboot.h
//  
//

#if !defined(__XBOXHIDBOOT_H__)
#define __XBOXHIDBOOT_H__

// Obsuga urzdzenie host USB wykorzystuje wiele plikw implementujcych wikszo niskopoziomowych funkcji
#include <inttypes.h>
#include <avr/pgmspace.h>
#include "avrpins.h"
#include "max3421e.h"
#include "usbhost.h"
#include "usb_ch9.h"
#include "Usb.h"
#include "hid.h"

#if defined(ARDUINO) && ARDUINO >=100
#include "Arduino.h"
#else
#include <WProgram.h>
#endif

#include "printhex.h"
#include "hexdump.h"
#include "message.h"

#include "confdescparser.h"

// Utworzenie struktury opisujcej format danych zwracanych przez urzdzenie USB
struct XBOXINFO
{
	struct
	{
        uint8_t bmReportType;
        uint8_t bmReportLength;	
        uint8_t bmPad;
        uint8_t bmButtons;
        uint8_t bmLeftTrigger;
        uint8_t bmRightTrigger;
    };
    
	int bmLeftStickXAxis;
	int bmLeftStickYAxis;
	int bmRightStickXAxis;
	int bmRightStickYAxis;	
};

// Ponisza klasa jest parserem raportw
class XboxReportParser : public HIDReportParser
{
    
protected:
	union 
	{
		XBOXINFO	xboxInfo;
		uint8_t		bInfo[sizeof(XBOXINFO)];
	}	prevState;
    
    
public:
	virtual void Parse(HID *hid, bool is_rpt_id, uint8_t len, uint8_t *buf);
	
protected:
	// Definicje funkcji obsugujcych przyciski kontrolera
	virtual void onDPadUp 			(XBOXINFO *mi) {};
	virtual void onDPadDown 		(XBOXINFO *mi) {};
	virtual void onDPadLeft 		(XBOXINFO *mi) {};
	virtual void onDPadRight 		(XBOXINFO *mi) {};
	virtual void onStartButton 		(XBOXINFO *mi) {};
	virtual void onBackButton 		(XBOXINFO *mi) {};
	virtual void onLeftStickPress 	(XBOXINFO *mi) {};
	virtual void onRightStickPress 	(XBOXINFO *mi) {};
	virtual void onButtonLB 		(XBOXINFO *mi) {};
	virtual void onButtonRB 		(XBOXINFO *mi) {};
	virtual void onButtonLogo 		(XBOXINFO *mi) {};
	virtual void onButtonA	 		(XBOXINFO *mi) {};
	virtual void onButtonB	 		(XBOXINFO *mi) {};
	virtual void onButtonX	 		(XBOXINFO *mi) {};
	virtual void onButtonY	 		(XBOXINFO *mi) {};
	virtual void onLeftTrigger		(XBOXINFO *mi) {};
	virtual void onRightTrigger		(XBOXINFO *mi) {};
	virtual void onLeftStickMove	(XBOXINFO *mi) {};
	virtual void onRightStickMove	(XBOXINFO *mi) {};
	
};

// Konfiguracja urzdzenia USB
template <const uint8_t BOOT_PROTOCOL>
class HIDBoot : public HID // UsbConfigXtracter
{
	EpInfo		epInfo[totalEndpoints];
    
	HIDReportParser	*pRptParser;
    
	uint8_t		bConfNum;       // Numer konfiguracji
	uint8_t		bIfaceNum;      // Numer interfejsu
	uint8_t		bNumIface;      // Ilo interfejsw w konfiguracji
	uint8_t		bNumEP;         // czna ilo EP w konfiguracji
	uint32_t	qNextPollTime;  // Nastpny czas odczytu 
	bool		bPollEnable;    // Flaga wczonego odczytu
	
    void Initialize();          // Inicjalizacja funkcji urzdzenia
    
	virtual HIDReportParser* GetReportParser(uint8_t id) { return pRptParser; };
    
public:
	HIDBoot(USB *p);
    
	virtual bool SetReportParser(uint8_t id, HIDReportParser *prs) { pRptParser = prs; };
    
	virtual uint8_t Init(uint8_t parent, uint8_t port, bool lowspeed); // Inicjalizacja klasy USBDeviceConfig
	virtual uint8_t Release();
	virtual uint8_t Poll();
	virtual uint8_t GetAddress() { return bAddress; };
    
	
	virtual void EndpointXtract(uint8_t conf, uint8_t iface, uint8_t alt, uint8_t proto, const USB_ENDPOINT_DESCRIPTOR *ep); // Implementacja klasy UsbConfigXtracter
};

template <const uint8_t BOOT_PROTOCOL>
HIDBoot<BOOT_PROTOCOL>::HIDBoot(USB *p) :
HID(p),
qNextPollTime(0),
bPollEnable(false),
pRptParser(NULL) // Ustawienie parametrw USB
{
	Initialize();
    
	if (pUsb)
		pUsb->RegisterDeviceClass(this);
}

// Implementacja funkcji inicjujcej urzdzenie
template <const uint8_t BOOT_PROTOCOL>
void HIDBoot<BOOT_PROTOCOL>::Initialize()
{
	for(uint8_t i=0; i<totalEndpoints; i++)
	{
		epInfo[i].epAddr		= 0;
		epInfo[i].maxPktSize	= (i) ? 0 : 20;
		epInfo[i].epAttribs		= 0;
		epInfo[i].bmNakPower	= (i) ? USB_NAK_NOWAIT : USB_NAK_MAX_POWER;
	}
	bNumEP		= 1;
	bNumIface	= 0;
	bConfNum	= 0;
}

// Implementacja inicjalizacji urzdzenia
template <const uint8_t BOOT_PROTOCOL>
uint8_t HIDBoot<BOOT_PROTOCOL>::Init(uint8_t parent, uint8_t port, bool lowspeed)
{
	const uint8_t constBufSize = sizeof(USB_DEVICE_DESCRIPTOR);
    
	// Deklaracje zmiennych urzdzenia USB
	uint8_t		buf[constBufSize];
	uint8_t		rcode; 
	UsbDevice	*p = NULL; 
	EpInfo		*oldep_ptr = NULL;
	uint8_t		len = 0;
	uint16_t	cd_len = 0;
	uint8_t		num_of_conf;
	uint8_t		num_of_intf;
    
	AddressPool	&addrPool = pUsb->GetAddressPool();
    
	p = addrPool.GetUsbDevicePtr(0); // Pobranie wskanika do pseudo-urzdzenia o adresie 0
    
	oldep_ptr = p->epinfo; // Zapisanie starego wskanika rekordu EP_RECORD o adresie 0
    
	p->epinfo = epInfo; // Tymczasowe przypisanie nowego wskanika do epInfo do p->epinfo w celu uniknicia niespjnoci
    
	p->lowspeed = lowspeed;
    
	rcode = pUsb->getDevDescr( 0, 0, 8, (uint8_t*)buf ); // Pobranie deskryptora urzdzenia
    
	if  (!rcode)
		len = (buf[0] > constBufSize) ? constBufSize : buf[0];
    
    p->epinfo = oldep_ptr; // Odtworzenie p->epinfo
    
    bAddress = addrPool.AllocAddress(parent, false, port); // Alokacja nowego adresu waciwego la klasy urzdzenia
    
    epInfo[0].maxPktSize = (uint8_t)((USB_DEVICE_DESCRIPTOR*)buf)->bMaxPacketSize0; // Wyodrbnienie maksymalnej wielkoci pakietu z deskryptora urzdzenia
    
    rcode = pUsb->setAddr( 0, 0, bAddress ); // Przypisanie nowego adresu do urzdzenia
    
    p->lowspeed = false;
    
    p = addrPool.GetUsbDevicePtr(bAddress);
    
    p->lowspeed = lowspeed;
    
    if (len)
        rcode = pUsb->getDevDescr( bAddress, 0, len, (uint8_t*)buf );
    
    num_of_conf = ((USB_DEVICE_DESCRIPTOR*)buf)->bNumConfigurations;
    
    rcode = pUsb->setEpInfoEntry(bAddress, 1, epInfo); // Przypisanie epInfo do wskanika epInfo
    
    for (uint8_t i=0; i<num_of_conf; i++)
    {
        ConfigDescParser<
        USB_CLASS_HID, 
        HID_BOOT_INTF_SUBCLASS, 
        BOOT_PROTOCOL, 
        CP_MASK_COMPARE_PROTOCOL> confDescrParser(this);
        
        rcode = pUsb->getConfDescr(bAddress, 0, i, &confDescrParser);
        
        if (bNumEP > 1)
            break;
    } 
    
	
	rcode = pUsb->setEpInfoEntry(bAddress, bNumEP, epInfo); // Przypisanie epInfo do wskanika epInfo
    
	rcode = pUsb->setConf(bAddress, 0, bConfNum); // Ustawienie numeru konfiguracji
    
	
	USBTRACE("XBOX skonfigurowany\r\n");
	bPollEnable = true;
	return 0;
    
}

// Wyodrbnienie punktu kocowego
template <const uint8_t BOOT_PROTOCOL>
void HIDBoot<BOOT_PROTOCOL>::EndpointXtract(uint8_t conf, uint8_t iface, uint8_t alt, uint8_t proto, const USB_ENDPOINT_DESCRIPTOR *pep) 
{
	if (bNumEP > 1 && conf != bConfNum)
		return;
    
	bConfNum	= conf;
	bIfaceNum	= iface;
    
	uint8_t index;
    
	if ((pep->bmAttributes & 0x03) == 3 && (pep->bEndpointAddress & 0x80) == 0x80)
	{
		index = epInterruptInIndex;
        
        // Wypenienie struktury informacjami o punkcie kocowym
		epInfo[index].epAddr		= (pep->bEndpointAddress & 0x0F);
		epInfo[index].maxPktSize	= (uint8_t)pep->wMaxPacketSize;
		epInfo[index].epAttribs		= 0;
        
		bNumEP ++;
	}
}

// Zwolnienie pamici po odczeniu urzdzenia
template <const uint8_t BOOT_PROTOCOL>
uint8_t HIDBoot<BOOT_PROTOCOL>::Release()
{
	pUsb->GetAddressPool().FreeAddress(bAddress);
    
	bConfNum			= 0;
	bIfaceNum			= 0;
	bNumEP				= 1;
	bAddress			= 0;
	qNextPollTime		= 0;
	bPollEnable			= false;
	return 0;
}

// Funkcja pobierajca dane z przerwania i sprawdzajca, czy s dane oczekujce na odczytanie
template <const uint8_t BOOT_PROTOCOL>
uint8_t HIDBoot<BOOT_PROTOCOL>::Poll()
{
	uint8_t rcode = 0;
    
	if (!bPollEnable)
		return 0;
    
	if (qNextPollTime <= millis())
	{
		qNextPollTime = millis() + 10;
        
		const uint8_t const_buff_len = 20;
		uint8_t buf[const_buff_len];
        
		uint16_t read = (uint16_t)epInfo[epInterruptInIndex].maxPktSize;
        
		uint8_t rcode = pUsb->inTransfer(bAddress, epInfo[epInterruptInIndex].epAddr, &read, buf);
        
		if (rcode)
		{
			if (rcode != hrNAK)
				USBTRACE2("Odczyt:", rcode);
            return rcode;
		}
		for (uint8_t i=0; i<read; i++)
			PrintHex<uint8_t>(buf[i]);
        if (read)
            Serial.println("");
        
		// Jeeli s dane, s odczytywane i analizowane w celu wyodrbnienia informacji o nacinitych
		// lub zmienionych przyciskach
        if (pRptParser)
            pRptParser->Parse((HID*)this, 0, (uint8_t)read, buf);
    }
	return rcode;
}


#endif
