/*
 * Szkic remote_decode
 * Sygnały IR pilota są odkodowywane w celu sterowania jasnością diody LED
 * Wartości przycisków 0-4 są odczytywane i zapisywane na początku szkicu
 * Przycisk 0 wyłącza diodę LED, a jasność jest stopniowo zwiększana za pomocą klawiszy od 1 do 4
 */

#include <IRremote.h>           // Biblioteka zdalnego sterowania na podczerwień

const int irReceivePin = 2;     // Pin wyjścia odbiornika IR
const int ledPin       = 9;     // Dioda LED podłączona do pinu PWM

const int numberOfKeys = 5;     // Szkic uczy się pięciu przycisków (od 0 do 4)
long irKeyCodes[numberOfKeys];  // Tablica przechowująca kod dla każdego z przycisków

IRrecv irrecv(irReceivePin);    // Utworzenie obiektu IRrecv
decode_results results;         // Zmienna przechowująca odczyty z detektora podczerwieni

void setup()
{
  Serial.begin(9600);
  while(!Serial); // Linia wymagana dla Leonardo i płytek ARM
  pinMode(irReceivePin, INPUT);
  pinMode(ledPin, OUTPUT);
  irrecv.enableIRIn();              // Uruchomienie odbiornika IR
  learnKeycodes();                  // Nauka kodów przycisków pilota
  Serial.println("Naciśnij przycisk na pilocie");
}

void loop()
{
  long key;
  int  brightness;

  if (irrecv.decode(&results))
  {
    // Poniższe operacje są wykonywane, jeśli otrzymano dane
    irrecv.resume();
    key = convertCodeToKey(results.value);
    if(key >= 0)
    {
      Serial.print("Naciśnięto przycisk ");
      Serial.println(key);
      brightness = map(key, 0,numberOfKeys-1, 0, 255);
      analogWrite(ledPin, brightness);
    }
  }
}

/*
 * Uzyskanie kodów przycisków pilota
 */
void learnKeycodes()
{
  while(irrecv.decode(&results))   // Wyczyszczenie bufora
    irrecv.resume();
 
  Serial.println("Gotowy do nauki przycisków pilota");
  long prevValue = -1;
  int i=0;
  while(i < numberOfKeys)
  {
    Serial.print("Naciśnij przycisk ");
    Serial.print(i);
    while(true)
    {
      if(irrecv.decode(&results))
      {
        if(results.value != -1 && 
           results.decode_type != UNKNOWN && 
           results.value != prevValue)
        {
          showReceivedData();
          Serial.println(results.value);
          irKeyCodes[i] = results.value;
          i = i + 1;
          prevValue = results.value;
          irrecv.resume(); // Odbieranie kolejnej wartości
          break;
        }
        irrecv.resume(); // Odbieranie kolejnej wartości
      }
    }
  }
  Serial.println("Nauka zakończona");
}

/*
 * Zamiana kodów protokołu pilota na logiczny kod przycisku
 * (-1 w przypadku nieotrzymania żadnej cyfry)
 */
int convertCodeToKey(long code)
{
  for(int i=0; i < numberOfKeys; i++)
  {
    if(code == irKeyCodes[i])
    {
      return i; // Zwracanie przycisku, jeśli został odnaleziony
    }
  }
  return -1;
}

/*
 * Wyświetlenie protokołu i jego wartości
 */
void showReceivedData()
{
  if (results.decode_type == UNKNOWN)
  {
    Serial.println("-Nie można odkodować wiadomości");
  }
  else
  {
    if (results.decode_type == NEC) {
      Serial.print("- odkodowany NEC: ");
    }
    else if (results.decode_type == SONY) {
      Serial.print("- odkodowany SONY: ");
    }
    else if (results.decode_type == RC5) {
      Serial.print("- odkodowany RC5: ");
    }
    else if (results.decode_type == RC6) {
      Serial.print("- odkodowany RC6: ");
    }
    Serial.print("wartość w zapisie szesnastkowym = ");
    Serial.println( results.value, HEX);
  }
}
