/*
 * Szkic inputcapture
 * Używa licznika do pomiaru impulsów na pinie 8 na ATmega 168/328
 */

/* Kilka ciekawych wzorów bitowych ASCII:
 u 01010101
 3 00110011
 ~ 01111110
 @ 01000000
*/ 

const int inputCapturePin = 8;     // Pin wejścia przypisany do wewnętrznego licznika
const int ledPin          = LED_BUILTIN;   

const int prescale = 8;            // Współczynnik dzielnika częstotliwości (każdy tik trwa 0,5 us przy częstotliwości 16 MHz)
const byte prescaleBits = B010;    // Zobacz tabelę 18.1 lub dokumenatcję mikrokontrolera
// Obliczenie czasu na tik licznika w nanosekundach
const long  precision = (1000000/(F_CPU/1000.0)) * prescale;   

const int numberOfEntries  = 64;    // Maksymalna liczba impulsów do zmierzenia
const int gateSamplePeriod = 1000;  // Próbny okres podany w milisekundach

volatile byte index = 0; // Indeks zapisanego odczytu
volatile byte gate  = 0; // 0 wyłącza pomiar, 1 włącza
volatile unsigned int results[numberOfEntries]; // Zauważ, że jest to wartość 16-bitowa

/* wektor przerwania ICR  */
ISR(TIMER1_CAPT_vect)
{
  TCNT1 = 0;                            // Wyzerowanie licznika
  if(gate)
  {
    if( index != 0 || bitRead(TCCR1B ,ICES1) == true)   // Oczekiwanie na wzrost sygnału
    {                                     // Został wykryty spadek sygnału
      if(index < numberOfEntries)
      {      
        results[index] = ICR1;              // Zapisanie wartości pomiaru wejścia
        index++;
      } 
    }
  }
  TCCR1B ^= _BV(ICES1);            // Zmiana bitu na przeciwny, aby pomiar był wywoływany przez drugą krawędź
}

void setup() {
  Serial.begin(9600);
  pinMode(ledPin, OUTPUT); 
  pinMode(inputCapturePin, INPUT); // Pin ICP (pin cyfrowy 8 na Arduino) jako wejście

  TCCR1A = 0 ;                    // Normalny tryb liczenia
  TCCR1B = prescaleBits ;         // Ustawienie bitów dzielnika częstotliwości
  TCCR1C = 0;
  bitSet(TCCR1B,ICES1);           // Inicjalizacja pomiaru wejścia
  bitSet(TIFR1,ICF1);             // Usunięcie oczekujących bitów  
  bitSet(TIMSK1,ICIE1);           // Włączenie  
   
  Serial.println("Gdy dioda LED się świeci, wykonywany jest pomiar impulsów.");
  Serial.print( precision);     // Wyświetlenie długości każdego tiku w mikrosekundach
  Serial.println(" mikrosekund na tik");
}

// Ta pętla wyświetla długość impulsów wykrytych w ostaniej sekundzie
void loop() 
{
  digitalWrite(ledPin, LOW);
  delay(gateSamplePeriod);
  digitalWrite(ledPin, HIGH);  
  index = 0;
  gate = 1; // Włączenie pomiaru
  delay(gateSamplePeriod);
  gate = 0;  // Wyłączenie pomiaru
  if(index > 0)
  {
    Serial.println("Czasy w mikrosekundach wynoszą:") ;
    for( byte i=0; i < index; i++)
    {
      long duration;
      duration = results[i] * precision; // Długość impulsu w nanosekundach
      if(duration > 0) {
        Serial.println(duration / 1000);   // Długość impulsu w mikrosekundach
        results[i] = 0; // Wyzerowanie wartości przed następnym odczytem
      }  
    }
    index = 0;
  }
}
