/*
 * Szkic eeprom
 * Program oparty na szkicu Blink bez funkcji delay
 * Używa pamięci EEPROM do przechowywania wartości, od których zależna jest częstotliwość migania
 */

#include <EEPROM.h>

// Wartości zapisywane w EEPROM
const byte EEPROM_ID = 0x99;   // Zmienna sprawdzająca czy dane w EEPROM są poprawne
byte ledPin =  LED_BUILTIN;    // Pin, do któego podłączona jest dioda LED
int interval = 1000;           // Interwał między mignięciami (w milisekundach)

// Zmienne, które nie muszą być zapisywane
int ledState = LOW;             // Zmienna ledState używana do ustawiania stanu LED
long previousMillis = 0;        // Zmienna przechowująca moment ostatniej zmiany stanu diody LED

//Stałe służące do identyfikacji adresów EEPROM
const int ID_ADDR = 0;       // Adres EEPROM, pod którym jest zapisane  ID
const int PIN_ADDR = 1;      // Adres EEPROM, pod którym jest zapisany numer pinu
const int INTERVAL_ADDR = 2; // Adres EEPROM, pod którym jest zapisana wartość interwału

void setup()
{
  Serial.begin(9600);
  byte id = EEPROM.read(ID_ADDR); // Odczyt pierwszego bajta z EEPROM
  if( id == EEPROM_ID)
  {
    // Jeśli odczytana wartość id zgadza się z zapisaną w EEPROM wartością, użyj danych z EEPROM
    Serial.println("Używanie danych z EEPROM");
    ledPin = EEPROM.read(PIN_ADDR);
    byte hiByte  = EEPROM.read(INTERVAL_ADDR);
    byte lowByte = EEPROM.read(INTERVAL_ADDR+1);
    interval =  word(hiByte, lowByte); // Funkcja word została opisana w recepturze 3.15
  }
  else
  {
    // Jeśli ID nie zostało odnalezione, zapisz dane domyślne
    Serial.println("Zapis danych domyślnych do EEPROM");
    EEPROM.write(ID_ADDR,EEPROM_ID); // Zapis ID w celu wskazania poprawnych danych
    EEPROM.write(PIN_ADDR, ledPin); // Zapis pinu w eeprom
    byte hiByte = highByte(interval);
    byte loByte = lowByte(interval);
    EEPROM.write(INTERVAL_ADDR, hiByte);
    EEPROM.write(INTERVAL_ADDR+1, loByte);
  }
  Serial.print("Ustawienie pinu na ");
  Serial.println(ledPin,DEC);
  Serial.print("Ustawienie interwału na ");
  Serial.println(interval);

  pinMode(ledPin, OUTPUT);
}

void loop()
{
  // Taki sam kod jak w szkicu BlinkWithoutDelay
  if (millis() - previousMillis > interval)
  {
    previousMillis = millis();     // Zapisanie informacji o czasie ostatniego mignięcia diodą
    // Jeśli dioda LED jest wyłączona, włącz ją i na odwrót
    if (ledState == LOW)
      ledState = HIGH;
    else
      ledState = LOW;
    digitalWrite(ledPin, ledState);  // Ustawienie stanu diody LED zgodnie z wartością zmiennej ledState
  }
  processSerial();
}

// funkcja pobierająca długość przerwy lub numeru pinu z monitora portu szeregowego
// wartość, po której jest umieszczona  litera i to przerwa, a p to numer pinu
int value = 0;
void processSerial()
{
   if( Serial.available())
  {
    char ch = Serial.read();
    if(ch >= '0' && ch <= '9') // Czy jest to cyfra w kodzie ASCII między 0 a 9?
    {
       value = (value * 10) + (ch - '0'); // Tak, dodaj ją do sumy poprzednich.
    }
    else if (ch == 'i')  // Czy to jest interwał?
    {
       interval = value;
       Serial.print("Ustawienie przerwy na ");
       Serial.println(interval);
       byte hiByte = highByte(interval);
       byte loByte = lowByte(interval);
       EEPROM.write(INTERVAL_ADDR, hiByte);
       EEPROM.write(INTERVAL_ADDR+1, loByte);
       value = 0; // wyzerowanie sumy dla następnej sekwencji liczb
    }
    else if (ch == 'p')  // Czy jest to numer pinu?
    {
       ledPin = value;
       Serial.print("Ustawienie pinu na ");
       Serial.println(ledPin,DEC);
       pinMode(ledPin, OUTPUT);
       EEPROM.write(PIN_ADDR, ledPin); // zapisanie pinu w EEPROM
       value = 0; // wyzerowanie sumy dla następnej sekwencji liczb
    }
  }
}
