#!/usr/bin/python
# -*- coding: utf-8 -*-
""" Moduł her_module """
from ansible.module_utils.basic import AnsibleModule

DOCUMENTATION = r'''
---
module: her_module
short_description: Sprawdzenie dostępności serwera
description: Moduł sprawdza, czy serwer jest dostępny

options:
  host:
    description:
      - Nazwa DNS lub adres IP
    required: true
  port:
    description:
      - Port TCP
      required: true
  timeout:
    description:
      - Maksymalny czas nawiązywania połączenia w sekundach
    required: false
    default: 3

requirements: [nmap]
author: Lorin Hochstein, Bas Meijer
notes:
  - To tylko przykład demonstrujący, jak napisać moduł
  - Prawdopodobnie zamiast tego można użyć natywnego modułu M(wait_for)
'''

EXAMPLES = r'''
# Sprawdzenie, czy SSH działa z domyślnym limitem czasu
- her_module: host=localhost port=22 timeout=1

# Sprawdzenie, czy PostreSQL działa z limitem czasu
- her_module: host=example.com port=5432
'''


def her_module(module, host, port, timeout):
    """ Metoda nawiązująca połączenie TCP za pomocą programu nc """
    nc_path = module.get_bin_path('nc', required=True)
    args = [nc_path, "-z", "-v", "-w", str(timeout), host, str(port)]
    return module.run_command(args, check_rc=True)


def main():
    """ Moduł nawiązujący połączenie za pomocą programu nc """
    module = AnsibleModule(
        argument_spec=dict(
            host=dict(required=True),
            port=dict(required=True, type='int'),
            timeout=dict(required=False, type='int', default=3)
        ),
        supports_check_mode=True
    )

    # W trybie weryfikacji nie wykonujemy żadnych operacji.
    # Ponieważ moduł nie zmienia stanu serwera, zwraca po prostu changed=False.

    if module.check_mode:
        module.exit_json(changed=False)
    host = module.params['host']
    port = module.params['port']
    timeout = module.params['timeout']

    if her_module(module, host, port, timeout)[0] == 0:
        msg = "Serwer %s:%s dostępny" % (host, port)
        module.exit_json(changed=False, msg=msg)
    else:
        msg = "Brak dostępu do serwera %s:%s" % (host, port)
        module.fail_json(failed=True, msg=msg)


if __name__ == "__main__":
    main()
