import { Component } from '@angular/core';
import {FormBuilder, FormControl, FormGroup, Validators} from '@angular/forms';

function ssnValidator(control: FormControl): {[key: string]: any} {
  const value: string = control.value || '';
  const valid = value.match(/^\d{9}$/);
  return valid ? null : {ssn: true};
}

function equalValidator({value}: FormGroup): {[key: string]: any} {
  const [first, ...rest] = Object.keys(value || {});
  const valid = rest.every(v => value[v] === value[first]);
  return valid ? null : {equal: true};
}

@Component({
  selector: 'app-root',
  template: `
    <form [formGroup]="formModel" (ngSubmit)="onSubmit()">
      <div>
        Nazwa użytkownika:
        <input type="text" formControlName="userName">
        <span class="error" [hidden]="!formModel.hasError('required', 'userName')">Nazwa użytkownika jest wymagana</span>
      </div>

      <div>
        Numer ubezpieczenia:
        <input type="text" formControlName="socialSecurity">
        <span class="error" [hidden]="!formModel.hasError('ssn', 'socialSecurity')">Numer ubezpieczenia jest nieprawidłowy</span>
      </div>

      <div formGroupName="passwordsGroup">
        <div>
          Hasło:
          <input type="password" formControlName="password">
          <span class="error" [hidden]="!formModel.hasError('minlength', ['passwordsGroup', 'password'])">Hasło jest za krótkie</span>
        </div>

        <div>
          Potwierdź hasło:
          <input type="password" formControlName="pconfirm">
          <span class="error" [hidden]="!formModel.hasError('equal', 'passwordsGroup')">Hasła muszą być takie same</span>
        </div>
      </div>

      <button type="submit" [disabled]="formModel.invalid">Wyślij</button>
    </form>
  `,
  styles: ['.error {color: red;} ']
})
export class AppComponent {
  formModel: FormGroup;

  constructor(fb: FormBuilder) {  
    this.formModel = fb.group({
    userName: ['', Validators.required], 
    socialSecurity: ['', ssnValidator],
    passwordsGroup: fb.group({
      password: ['', Validators.minLength(5)],
      pconfirm: ['']
    }, {validator: equalValidator}) 
   });
  }

  onSubmit() {
      console.log(this.formModel.value);
  }
}
