import {Component, Input, ViewEncapsulation, Inject, ElementRef, Output, EventEmitter, HostBinding} from '@angular/core';
import template from './tags-select.html!text';
import {TagsService} from '../tags-service';

@Component({
  selector: 'ngc-tags-select',
  host: {
    class: 'tags-select'
  },
  template,
  encapsulation: ViewEncapsulation.None
})
export class TagsSelect {
  // Tekst używany do filtracji dostępnych etykiet.
  @Input() filter;
  // Ogranicz wyświetlanie etykiet do podanego limitu.
  @Input() limit;
  // Obiekt wejściowy zawierający właściwości 'left' i 'top' pozwalające pozycjnonować komponent TagSelect we właściwym miejscu.
  @Input() position;
  // Właściwość wyjściowa wyemituje zdarzenie, jeśli wybrano jedną z prezentowanych etykiet.
  @Output() tagSelected = new EventEmitter();

  @HostBinding('style.display')
  get isVisible() {
    if (this.filter[0] === '#' && this.filteredTags.length > 0) {
      return 'block';
    } else {
      return 'none';
    }
  }

  @HostBinding('style.top')
  get topPosition() {
    return this.position ? `${this.position.top}px` : 0;
  }

  @HostBinding('style.left')
  get leftPosition() {
    return this.position ? `${this.position.left}px` : 0;
  }

  constructor(@Inject(TagsService) tagsService) {
    this.tagsService = tagsService;
    // Składowa przechowuje listę etykiet po filtracji.
    this.filteredTags = [];
    this.filter = '';
  }

  // Wykonaj poniższy kod po inicjalizacji właściwości wejściowych.
  ngOnInit() {
    // Usługa TagsService zapewnia nam wszystkie etykiety dostępne w aplikacji.
    this.tagsSubscription = this.tagsService.change.subscribe((tags) => {
      // Jeśli lista dostępnych etykiet ulegnie zmianie, zapamiętaj nową listę i przeprowadź ponowną filtrację.
      this.tags = tags;
      this.filterTags();
    });
  }

  // Jeśli kliknięto etykietę, wyemituj zdarzenie.
  onTagClick(tag) {
    this.tagSelected.next(tag);
  }

  // Funkcja filtruje wszystkie etykiety na podstawie kryterium filtracji z parametrów wejściowych i ogranicza liczbę wyników do zadanego limitu.
  filterTags() {
    const filter = this.filter.slice(1).toLowerCase();
    this.filteredTags = this.tags
      .filter((tag) => {
        return tag.textTag.indexOf(filter) !== -1 ||
          tag.title.toLowerCase().indexOf(filter) !== -1;
      })
      .slice(0, this.limit);
  }

  // Implementujemy tę funkcję, aby reagować na zmiany danych wejściowych.
  ngOnChanges(changes) {
    // Jeśli zmieni się parametr wejściowy filter lub limit, ponownie filtrujemy listę.
    if (this.tags && (changes.filter || changes.limit)) {
      this.filterTags();
    }
  }

  // Jeśli komponent jest niszczony, anuluj wszystkie subskrypcje, do których byliśmy zapisani.
  ngOnDestroy() {
    this.tagsSubscription.unsubscribe();
  }
}
