import {Component, ViewEncapsulation, ViewChild, ElementRef, Input} from '@angular/core';
import template from './tasks-chart.html!text';
import Chartist from 'chartist';
import Moment from 'moment';

import {rasterize} from '../../utilities/time-utilities';

@Component({
  selector: 'ngc-tasks-chart',
  host: {
    class: 'tasks-chart'
  },
  template,
  encapsulation: ViewEncapsulation.None
})
export class TasksChart {
  @Input() projects;
  @ViewChild('chartContainer') chartContainer;

  constructor() {
    // Zdefiniuj dostępne dla użytkownika zakresy czasu prezentowane na wykresie.
    this.timeFrames = [{
      name: 'dzień',
      timeFrame: 600000,
      amount: 144
    }, {
      name: 'tydzień',
      timeFrame: 3600000,
      amount: 168
    }, {
      name: 'rok',
      timeFrame: 86400000,
      amount: 365
    }];
    // Z dostępnych zakresów czasów pobieramy nazwy, aby przekazać je do komponentu przełącznika.
    this.timeFrameNames = this.timeFrames.map((timeFrame) => timeFrame.name);
    // Ustaw domyślny zakes na pierwszy z możliwych.
    this.selectedTimeFrame = this.timeFrames[0];
  }

  ngOnChanges() {
    if (this.projects) {
      // Po zmianie danych wejściowych musimy uaktualnić legendę.
      this.legend = this.projects.map((project, index) => {
        return {
          name: project.title,
          class: `tasks-chart__series--series-${index + 1}`
        };
      });
    }

    this.createOrUpdateChart();
  }

  ngAfterViewInit() {
    this.createOrUpdateChart();
  }

  // Wywoływane przez komponent przełącznika po wybraniu nowego zakresu czasu.
  onSelectedTimeFrameChange(timeFrameName) {
    // Ustaw wybrany zakres na wartość klikniętą przez użytkownika w komponencie toggle.
    this.selectedTimeFrame = this.timeFrames.find((timeFrame) => timeFrame.name === timeFrameName);
    this.createOrUpdateChart();
  }

  createOrUpdateChart() {
    if (!this.projects || !this.chartContainer) {
      return;
    }

    // Utwórz tablicę serii, która zawiera po jednej serii dla każdego projektu.
    const series = this.projects.map((project) => {
      // Zredukuj wszystkie zadania do jednej listy timeData.
      const timeData = project.tasks.reduce((timeData, task) => {
        // Czas utworzenia zadania generuje timeData z wagą 1.
        timeData.push({
          time: task.created,
          weight: 1
        });
        // Jeśli zadanie jest wykonane, generuje timeData z wagą -1.
        if (task.done) {
          timeData.push({
            time: task.done,
            weight: -1
          });
        }
        return timeData;
      }, []);

      // Wykorzystując funkcję rasterize() w trybie akumulacji, tworzymy odpowiednią tablicę danych reprezentującą serię.
      return rasterize(timeData, this.selectedTimeFrame.timeFrame, this.selectedTimeFrame.amount, +new Date(), null, true);
    });

    const now = +new Date();
    // Utwórz etykiety dla wszystkich wyświetlanych zakresów.
    const labels = Array.from({
      length: this.selectedTimeFrame.amount
    }).map((e, index) => now - index * this.selectedTimeFrame.timeFrame).reverse();

    if (this.chart) {
      // Jeśli mamy odpowiedni obiekt wykresu, wystarczy, że uaktualnimy serie i etykiety.
      this.chart.update({
        series,
        labels
      });
    } else {
      // Tworzenie nowego wykresu z elementem chartContainer jako kontener.
      this.chart = new Chartist.Line(this.chartContainer.nativeElement, {
        series,
        labels
      }, {
        width: '100%',
        height: 300,
        // Dzięki interpolacji krokowej wykres będzie renderowany z krokami zamiast z bezpośrednio połączonymi punktami.
        lineSmooth: Chartist.Interpolation.step({
          // Opcja ta włączona dla interpolacji spowoduje pomijanie wartości null i połączenie linii do następnej poprawnej wartości.
          fillHoles: true
        }),
        axisY: {
          onlyInteger: true,
          low: 0,
          offset: 70,
          // Używamy funkcji interpolacji etykiet do formatowania liczby otwartych zadań.
          labelInterpolationFnc: (value) => (value === 1 ? `${value} zadanie` : value >= 2 && value <= 4 ? `${value} zadania` : `${value} zadań`)
        },
        axisX: {
          // Na małych ekranach wyświetlamy jedynie dwie etykiety na osi X i linie siatki.
          labelInterpolationFnc: (value, index, array) => index % Math.floor(this.selectedTimeFrame.amount / 2) === 0 ? Moment(value).calendar() : null
        }
      }, [
        ['screen and (min-width: 1200px)', {
          // Na dużych ekranach wyświetlamy cztery etykiety na osi X i linie siatki.
          axisX: {
            labelInterpolationFnc: (value, index) => index % Math.floor(this.selectedTimeFrame.amount / 4) === 0 ? Moment(value).calendar() : null
          }
        }], ['screen and (min-width: 1500px)', {
          // Na bardzo dużych ekranach wyświetlamy sześć etykiet na osi X i linie siatki.
          axisX: {
            labelInterpolationFnc: (value, index) => index % Math.floor(this.selectedTimeFrame.amount / 6) === 0 ? Moment(value).calendar() : null
          }
        }]
      ]);
    }
  }
}
