import {inject, async, TestBed} from '@angular/core/testing';
import {By} from '@angular/platform-browser';
import {Component, Input, Injectable, Compiler} from '@angular/core';
import {Subject} from 'rxjs/Rx';
import {PluginConfig, PluginPlacement} from './plugin';
import {PluginService} from './plugin-service'
import {PluginSlot} from './plugin-slot';
import {NgModule} from '@angular/core';

@Injectable()
export class MockPluginService extends PluginService {
  constructor() {
    super();
    this.change = {
      subscribe() {
        return jasmine.createSpyObj("change", ["unsubscribe"]);
      }
    };
  }

  loadPlugins() {}
}

@Component({
  selector: 'dummy-plugin-component-1',
  template: 'dummy1'
})
export class DummyPluginComponent1 {}

@Component({
  selector: 'dummy-plugin-component-2',
  template: 'dummy2'
})
export class DummyPluginComponent2 {}

@NgModule({
  declarations: [DummyPluginComponent1, DummyPluginComponent2]
})
class DummyPluginModule {}

@Component({
  selector: 'dummy-application',
  template: 'dummy-slot:<ngc-plugin-slot name="dummy-slot"></ngc-plugin-slot>'
})
export class DummyApplication {}

describe('PluginSlot', () => {
  beforeEach(() => {
    TestBed.configureTestingModule({
      declarations: [DummyApplication, PluginSlot],
      providers: [{provide: PluginService, useClass: MockPluginService}]
    });
  });

  afterEach(() => {
    TestBed.resetTestingModule();
  });

  it('powinien utworzyć testowy komponent w odpowiednim miejscu', async(inject([PluginService, Compiler], (pluginService, compiler) => {
    TestBed.compileComponents().then(() => {
      const fixture = TestBed.createComponent(DummyApplication);
      fixture.detectChanges();
      expect(fixture.nativeElement.textContent).toBe('dummy-slot:');

      @PluginConfig({
        name: 'dummy-plugin',
        description: 'Testowy plug-in',
        module: DummyPluginModule,
        placements: [
          new PluginPlacement({slot: 'dummy-slot', priority: 1, component: DummyPluginComponent1})
        ]
      })
      class DummyPlugin {}

      pluginService.plugins = [{
        type: DummyPlugin,
        config: DummyPlugin._pluginConfig,
        instance: new DummyPlugin(),
        componentFactories: compiler.compileModuleAndAllComponentsSync(DummyPlugin._pluginConfig.module).componentFactories
      }];

      const pluginSlot = fixture.debugElement
        .query(By.directive(PluginSlot))
        .injector
        .get(PluginSlot);

      pluginSlot.initialize();

      fixture.detectChanges();
      expect(fixture.nativeElement.textContent).toBe('dummy-slot:dummy1');
    });
  })));

  it('powinien utworzyć dwa testowe komponenty tego samego plug-inu w różnych miejscach',
    async(() => {
      const template = 'dummy-slot1:<ngc-plugin-slot name="dummy-slot1"></ngc-plugin-slot>dummy-slot2:<ngc-plugin-slot name="dummy-slot2"></ngc-plugin-slot>';
      TestBed.overrideComponent(DummyApplication, {
        set: {
          template: template
        }
      });

      inject([PluginService, Compiler], (pluginService, compiler) => {
        TestBed.compileComponents().then(() => {
          const fixture = TestBed.createComponent(DummyApplication);
          fixture.detectChanges();
          expect(fixture.nativeElement.textContent).toBe('dummy-slot1:dummy-slot2:');

          @PluginConfig({
            name: 'dummy-plugin',
            description: 'Testowy plug-in',
            module: DummyPluginModule,
            placements: [
              new PluginPlacement({slot: 'dummy-slot1', priority: 1, component: DummyPluginComponent1}),
              new PluginPlacement({slot: 'dummy-slot2', priority: 1, component: DummyPluginComponent2})
            ]
          })
          class DummyPlugin {
          }

          pluginService.plugins = [{
            type: DummyPlugin,
            config: DummyPlugin._pluginConfig,
            instance: new DummyPlugin(),
            componentFactories: compiler.compileModuleAndAllComponentsSync(DummyPlugin._pluginConfig.module).componentFactories
          }];

          const pluginSlots = fixture.debugElement
            .queryAll(By.directive(PluginSlot))
            .map((debugElement) => debugElement.injector.get(PluginSlot));

          pluginSlots.forEach((pluginSlot) => pluginSlot.initialize());

          fixture.detectChanges();
          expect(fixture.nativeElement.textContent).toBe('dummy-slot1:dummy1dummy-slot2:dummy2');
        });
      })();
    })
  );

  it('powinien utworzyć dwa komponenty tego samego pluginu w tym samym miejscu z uwzględnieniem priorytetów',
    async(inject([PluginService, Compiler], (pluginService, compiler) => {
      TestBed.compileComponents().then(() => {
        const fixture = TestBed.createComponent(DummyApplication);
        fixture.detectChanges();
        expect(fixture.nativeElement.textContent).toBe('dummy-slot:');

        @PluginConfig({
          name: 'dummy-plugin',
          description: 'Testowy plug-in',
          module: DummyPluginModule,
          placements: [
            new PluginPlacement({slot: 'dummy-slot', priority: 1, component: DummyPluginComponent1}),
            new PluginPlacement({slot: 'dummy-slot', priority: 2, component: DummyPluginComponent2})
          ]
        })
        class DummyPlugin {}

        pluginService.plugins = [{
          type: DummyPlugin,
          config: DummyPlugin._pluginConfig,
          instance: new DummyPlugin(),
          componentFactories: compiler.compileModuleAndAllComponentsSync(DummyPlugin._pluginConfig.module).componentFactories
        }];

        const pluginSlot = fixture.debugElement
          .query(By.directive(PluginSlot))
          .injector
          .get(PluginSlot);

        pluginSlot.initialize();

        fixture.detectChanges();
        expect(fixture.nativeElement.textContent).toBe('dummy-slot:dummy2dummy1');
      });
    })));

});
