import {Component, Inject, Input, Output, ViewEncapsulation, ViewChild, EventEmitter} from '@angular/core';
import template from './comments.html!text';
import {UserService} from '../user/user-service/user-service';

@Component({
  selector: 'ngc-comments',
  host: {
    class: 'comments'
  },
  template,
  encapsulation: ViewEncapsulation.None
})
export class Comments {
  // Lista obiektów komentarza.
  @Input() comments;
  // Zdarzenie dotyczące aktualizacji listy komentarzy
  @Output() commentsUpdated = new EventEmitter();
  // Używamy edytora do dodawania nowych komentarzy, więc potrzebujemy referencji do tego komponentu.
  @ViewChild('newCommentEditor') newCommentEditor;

  // Używamy usługi użytkownika, aby pobrać dane aktualnie zalogowanego użytkownika.
  constructor(@Inject(UserService) userService) {
    this.userService = userService;
  }

  // Śledzimy zmiany danych wejściowych, aby zapobiec błędowi niezdefiniowanej listy komentarzy.
  ngOnChanges(changes) {
    if (changes.comments && changes.comments.currentValue === undefined) {
      this.comments = [];
    }
  }

  // Dodawanie nowego komentarza z pola newCommentEditor, które jest powiązane z edytorem.
  addNewComment() {
    const comments = this.comments.slice();
    comments.splice(0, 0, {
      user: this.userService.currentUser,
      time: +new Date(),
      content: this.newCommentEditor.getEditableContent()
    });
    // Wyemituj zdarzenie, aby uaktualniona lista komentarzy została trwale zapisana poza komponentem.
    this.commentsUpdated.next(comments);
    // Wyczyść zawartość edytora.
    this.newCommentEditor.setEditableContent('');
  }

  // Metoda obsługuje edytowane komentarze.
  onCommentEdited(comment, content) {
    const comments = this.comments.slice();
    // Jeśli po edycji komentarz ma zerową długość, usuń go z listy komentarzy.
    if (content.length === 0) {
      comments.splice(comments.indexOf(comment), 1);
    } else {
      // W przeciwnym razie zamień treść na nową.
      comments.splice(comments.indexOf(comment), 1, {
        user: comment.user,
        time: comment.time,
        content
      });
    }
    // Wyemituj zdarzenie, aby uaktualniony komentarz została trwale zapisany poza komponentem.
    this.commentsUpdated.next(comments);
  }

  hasComments() {
    return this.comments && this.comments.length > 0;
  }
}
