import {Injectable, Inject, Compiler} from '@angular/core';
import {PluginData} from './plugin';
import {ReplaySubject} from 'rxjs/Rx';

@Injectable()
export class PluginService {
  constructor(@Inject(Compiler) compiler) {
    this.compiler = compiler;
    this.plugins = [];
    // Umożliwienie obserwacji zmian listy aktywnych plug-inów.
    this.change = new ReplaySubject(1);
    this.loadPlugins();
  }

  // Funkcja wczytuje wszystkie plug-iny zdefiniowane w głównym pliku plugins.js.
  loadPlugins() {
    System.import('/plugins.js').then((pluginsModule) => {
      pluginsModule.default.forEach((pluginUrl) => this.loadPlugin(pluginUrl));
    });
  }

  // Wczytuje pojedynczy plug-in, tworzy jego egzemplarz i zapewnia mu informacje wykonawcze.
  loadPlugin(url) {
    return System.import(url).then((pluginModule) => {
      const Plugin = pluginModule.default;

      this.compiler.compileModuleAndAllComponentsAsync(Plugin._pluginConfig.module)
        .then(({componentFactories}) => {
          const pluginData = {
            url,
            type: Plugin,
            // Odczytanie danych umieszczonych wcześniej przez dekorator @Plugin
            config: Plugin._pluginConfig,
            // Utworzenie egzemplarza plug-inu.
            instance: new Plugin(),
            // Zapamiętaj fabryki komponentów, bo będą potrzebne do tworzenia pojedynczych komponentów.
            componentFactories: componentFactories
          };

          this.plugins = this.plugins.concat([pluginData]);
          this.change.next(this.plugins);
        });
    });
  }

  // Usunięcie plug-inu z listy.
  removePlugin(name) {
    const plugin = this.plugins.find((plugin) => plugin.name === name);
    if (plugin) {
      // Jeśli plug-in został znaleziony po nazwie, usuwamy go z listy i emitujemy zdarzenie.
      const plugins = this.plugins.slice();
      plugins.splice(plugins.indexOf(plugin), 1);
      this.plugins = plugins;
      this.change.next(this.plugins);
    }
  }

  // Pobiera dane wszystkich plug-inów, które chcą coś wstawić we wskazanym miejscu.
  getPluginData(slot) {
    return this.plugins.reduce((components, pluginData) => {
      return components.concat(
        pluginData.config.placements
          .filter((placement) => placement.slot === slot)
          .map((placement) => new PluginData(pluginData, placement))
      );
    }, []);
  }
}
