import {Input, Inject, Directive, ChangeDetectorRef, IterableDiffer, IterableDiffers, ViewContainerRef, TemplateRef, EmbeddedViewRef} from '@angular/core';

// ta funkcja pomocnicza przeszukuje drzewo DOM w poszukiwaniu następnego, wyższego elementu,
// który ma wartość overflowY inną niż visible.
function findScrollableParent(element) {
  while (element != document.documentElement) {
    if (getComputedStyle(element).overflowY !== 'visible') {
      break;
    }
    element = element.parentElement;
  }

  return element;
}

// Sprawdza, czy element z paskami przewijania jest w pełni przesunięty w dół.
function isScrolledBottom(element) {
  // Dodatkowe zabezpieczenie, gdyby scrollTop zwracało wartość ułamkową.
  return Math.abs(element.scrollHeight - element.scrollTop - element.clientHeight) < 1;
}

@Directive({
  selector: '[ngcInfiniteScroll]'
})
export class InfiniteScroll {
  constructor(@Inject(ViewContainerRef) viewContainerRef,
              @Inject(TemplateRef) templateRef,
              @Inject(IterableDiffers) iterableDiffers,
              @Inject(ChangeDetectorRef) cdr) {
    // Stosując Object.assign możemy łatwo dodać wszystkie argumenty konstruktora do instancji.
    Object.assign(this, {viewContainerRef, templateRef, iterableDiffers, cdr});
    // Ile elementów pokażemy początkowo.
    this.shownItemCount = 3;
    // Ile elementów należy pokazać dodatkowo, gdy przesuniemy się na dół.
    this.increment = 3;
  }

  ngOnInit() {
    // Zapamiętujemy pierwszy element nadrzędny, który można przewijać na osi Y.
    this.scrollableElement = findScrollableParent(this.viewContainerRef.element.nativeElement.parentElement);
    this._onScrollListener = this.onScroll.bind(this);
    // Jeśli element nadrzędny wygeneruje zdarzenie scroll, wywołaj metodę onScroll().
    this.scrollableElement.addEventListener('scroll', this._onScrollListener);
  }

  // Tę wartość wejściową ustawi składnia dotycząca pętli for-of.
  @Input('ngcInfiniteScrollOf')
  set infiniteScrollOfSetter(value) {
    this.infiniteScrollOf = value;
    // Utwórz obiekt differ dla 'value', jeśli jeszcze nie istnieje.
    if (value && !this.differ) {
      this.differ = this.iterableDiffers.find(value).create(this.cdr);
    }
  }

  // Metoda powinna być wywoływana po zajściu zdarzenia przewijania w elemencie nadrzędnym.
  onScroll() {
    // Jeśli element nadrzędny dotarł do końca paska przewijania, zwiększamy liczbę wyświetlanych elementów.
    if (this.scrollableElement && isScrolledBottom(this.scrollableElement)) {
      this.shownItemCount = Math.min(this.infiniteScrollOf.length, this.shownItemCount + this.increment);
      // Po zwiększeniu liczby elementów, poinformuj mechanizm wykrywania zmian o konieczności przeliczenia.
      this.cdr.markForCheck();
      this.cdr.detectChanges();
    }
  }

  // Implementując tę funkcję cyklu życia komponentu, mierzemy odpowiedzialność za samodzielne wykrywanie zmian.
  ngDoCheck() {
    // Sprawdź, czy udało się ustawić element differ w `infiniteScrollOfSetter`.
    if (this.differ) {
      // Tworzymy nowy wycinek na podstawie liczby wyświetlonych elementów.
      // Następnie tworzymy obiekt changes zawierający różnicę wykorzystując przy tym IterableDiffer.
      const updatedList = this.infiniteScrollOf.slice(0, this.shownItemCount);
      const changes = this.differ.diff(updatedList);
      if (changes) {
        // Jeśli wystąpiły zmiany, wywołaj metodę applyChanges().
        this.applyChanges(changes);
        // Sprawdź ponownie, czy nie trzeba jeszcze raz zwiększyć liczby elementów.
        this.onScroll();
      }
    }
  }

  // Metoda pobiera obiekt changes z `IterableDiffer` i obsługuje niezbędną aktualizację DOM.
  applyChanges(changes) {
    // Najpierw utwórz listę rekordów ze wszystkimi przesunięciami i usunięciami.
    const recordViewTuples = [];
    changes.forEachRemovedItem((removedRecord) => recordViewTuples.push({record: removedRecord}));
    changes.forEachMovedItem((movedRecord) => recordViewTuples.push({record: movedRecord}));

    // Możemy hurtowo usunąć wszystkie przesunięte i usunięte widoki, co pozostawia nam tylko widoki przesunięte.
    const insertTuples = this.bulkRemove(recordViewTuples);
    // Poza przesunietymi rekordami dodajemy również wszystkie nowe rekordy.
    changes.forEachAddedItem((addedRecord) => insertTuples.push({record: addedRecord}));

    // Gdy mamy wszystkie przesunięte i dodane rekordy w 'insertTuples', stosujemy hurtowe wstawienie.
    // W wyniku otrzymujemy listę nowych widoków.
    // W tych widokach tworzymy lokalną zmienną '$implicit, która dowiąże elementy listy do zmiennej używanej w skłądni for-of.
    this.bulkInsert(insertTuples).forEach((tuple) =>
      tuple.view.context.$implicit = tuple.record.item);
  }

  // Metoda odłączy przesunięte rekordy od kontenera widoków.
  // Usunięte rekordy są całkowicie usuwane również z kontenera.
  bulkRemove(tuples) {
    tuples.sort((a, b) => a.record.previousIndex - b.record.previousIndex);
    // Redukcja rekordów zmiany w taki sposób, aby pozostały tylko przesunięcia.
    return tuples.reduceRight((movedTuples, tuple) => {
      // Jeśli indeks istnieje w rekordzie zmiany, oznacza to przesunięcie.
      if (tuple.record.currentIndex != null) {
        // Dla przesuniętych rekordów, jedynie odłączamy je od kontenera i przekazujemy do listy przesuniętych.
        tuple.view = this.viewContainerRef.detach(tuple.record.previousIndex);
        movedTuples.push(tuple);
      } else {
        // Jeśli to rekord usunięty, całkowicie usuwamy widok.
        this.viewContainerRef.remove(tuple.record.previousIndex);
      }
      return movedTuples;
    }, []);
  }

  // Metoda służy do hurtowego wstawienia przesuniętych rekordów i dodania całkowicie nowych.
  bulkInsert(tuples) {
    tuples.sort((a, b) => a.record.currentIndex - b.record.currentIndex);
    tuples.forEach((tuple) => {
      // Jeśli w rekordzie zmiany znajduje się widok, mamy do czynienia z przesunięciem.
      if (tuple.view) {
        // Ponownie wstawiamy odłączony widok do kontenera, ale w nowym miejscu.
        this.viewContainerRef.insert(tuple.view, tuple.record.currentIndex);
      } else {
        // Jeśli to całkowicie nowy rekord, tworzymy nowy widok i zapamiętujemy go we właściwości.
        tuple.view =
          this.viewContainerRef.createEmbeddedView(this.templateRef, {}, tuple.record.currentIndex);
      }
    });
    return tuples;
  }

  ngOnDestroy() {
    this.scrollableElement.removeEventListener('scroll', this._onScrollListener);
  }
}
