import {Directive, HostListener, Input, HostBinding, Output, EventEmitter} from '@angular/core';

@Directive({
  selector: '[draggableDropZone]'
})
export class DraggableDropZone {
  // Parametr określa, jakim typem danych ma być zainteresowany element umożliwiający upuszczenie przeciąganej treści.
  @Input() dropAcceptType;
  // Jeśli coś zostanie umuszczone w "strefie zrzutu", powinniśmy wyemitować zdarzenie.
  @Output() dropDraggable = new EventEmitter();
  // Dyrektywa dodaje klasę do elementu hostującego, jeśli nad strefą znajdzie się element właściwego typu.
  @HostBinding('class.draggable--over') over;

  constructor() {
    // Potrzebujemy tego licznika, aby wiedzieć, czy przeciągany obiekt nadal znajduje sięnad "strefą zrzutu".
    this.dragEnterCount = 0;
  }


  // Sprawdź, czy przeciągany obiekt jest odpowiedniego typu, odczytując dane draggable-type.
  typeIsAccepted(event) {
    const draggableType = Array.from(event.dataTransfer.types).find((key) =>
      key.indexOf('draggable-type') === 0);
    return draggableType && draggableType.split(':')[1] === this.dropAcceptType;
  }


  // Obsługa zdarzenia dragover na elemencie hostującym.
  @HostListener('dragover', ['$event'])
  onDragOver(event) {
    // Obsługa zadarzenia tylko wtedy, gdy przeciągany element jest akceptowany przez "strefę zrzutu".
    if (this.typeIsAccepted(event)) {
      // Zapobiegnij domyślnej akcji przeciągnięcia wbudowanej w przeglądarkę i ustaw właściwość dropEffect obiektu dataTransfer.
      event.preventDefault();
      event.dataTransfer.dropEffect = 'move';
    }
  }

  // Obsługa zdarzenia drop, które zajdzie w momencie umuszczania przeciąganego elementu.
  @HostListener('drop', ['$event'])
  onDrop(event) {
    // Obsługa zadarzenia tylko wtedy, gdy przeciągany element jest akceptowany przez "strefę zrzutu".
    if (this.typeIsAccepted(event)) {
      // Obsłuż obiekt z danymi otrzymany jako część zdarzenia.
      const data = JSON.parse(event.dataTransfer.getData('application/json'));
      // Po udanym pobraniu przeciągniętych danych, możemy przywrócić stan domyślny i zgłosić zdarzenie.
      this.over = false;
      this.dragEnterCount = 0;
      this.dropDraggable.next(data);
    }
  }

  // Obsługa zdarzenia enter wywoływanego po wejściu przeciąganego obiektu do "strefy zrzutu".
  @HostListener('dragenter', ['$event'])
  onDragEnter(event) {
    // Obsługa zadarzenia tylko wtedy, gdy przeciągany element jest akceptowany przez "strefę zrzutu".
    if (this.typeIsAccepted(event)) {
      this.over = true;
      // Używamy licznika do określenia, czy utrata przeciągania wynika z przejścia do elementu podrzędnego, czy wyjścia z głównego elementu.
      this.dragEnterCount++;
    }
  }

  // Obsługa zdarzenia dragleave wywoływanego po wyjściu przeciąganego obiektu ze "strefy zrzutu".
  @HostListener('dragleave', ['$event'])
  onDragLeave(event) {
    // Wykorzystując dragEnterCount wiemy, czy zdarzenie dragleave wynika z przejścia nad element potomny, czy z wyjścia poza strefę.
    if (this.typeIsAccepted(event) && --this.dragEnterCount === 0) {
      this.over = false;
    }
  }
}
