import {Component, Inject, Input, Output, ViewEncapsulation, ViewChild, EventEmitter} from '@angular/core';
import template from './comments.html!text';
import {UserService} from '../user/user-service/user-service';
import {ActivityService} from '../activities/activity-service/activity-service';
import {limitWithEllipsis} from '../utilities/string-utilities';

@Component({
  selector: 'ngc-comments',
  host: {
    class: 'comments'
  },
  template,
  encapsulation: ViewEncapsulation.None
})
export class Comments {
  // Lista obiektów komentarza.
  @Input() comments;
  // Obiekt do logowania aktywności.
  @Input() activitySubject;
  // Zdarzenie dotyczące aktualizacji listy komentarzy
  @Output() commentsUpdated = new EventEmitter();
  // Używamy edytora do dodawania nowych komentarzy, więc potrzebujemy referencji do tego komponentu.
  @ViewChild('newCommentEditor') newCommentEditor;

  // Używamy usługi użytkownika, aby pobrać dane aktualnie zalogowanego użytkownika.
  constructor(@Inject(UserService) userService, @Inject(ActivityService) activityService) {
    this.userService = userService;
    this.activityService = activityService;
  }

  // Śledzimy zmiany danych wejściowych, aby zapobiec błędowi niezdefiniowanej listy komentarzy.
  ngOnChanges(changes) {
    if (changes.comments && changes.comments.currentValue === undefined) {
      this.comments = [];
    }
  }

  // Dodawanie nowego komentarza z pola newCommentEditor, które jest powiązane z edytorem.
  addNewComment() {
    const comments = this.comments.slice();
    const content = this.newCommentEditor.getEditableContent();
    comments.splice(0, 0, {
      user: this.userService.currentUser,
      time: +new Date(),
      content: content
    });
    // Wyemituj zdarzenie, aby uaktualniona lista komentarzy została trwale zapisana poza komponentem.
    this.commentsUpdated.next(comments);
    // Wyczyść zawartość edytora.
    this.newCommentEditor.setEditableContent('');
    // Tworzenie aktywności dla dodawania komentarza.
    this.activityService.logActivity(
      this.activitySubject.id,
      'comments',
      'Dodano nowy komentarz',
      `Dodano nowy komentarz "${limitWithEllipsis(content, 30)}" do #${this.activitySubject.document.data._id}.`
    );
  }

  // Metoda obsługuje edytowane komentarze.
  onCommentEdited(comment, content) {
    const comments = this.comments.slice();
    // Jeśli po edycji komentarz ma zerową długość, usuń go z listy komentarzy.
    if (content.length === 0) {
      const removed = comments.splice(comments.indexOf(comment), 1)[0];
      // Tworzenie aktywności dla usunięcia komentarza.
      this.activityService.logActivity(
        this.activitySubject.id,
        'comments',
        'Usunięto komentarz',
        `Usunięto komentarz "${limitWithEllipsis(removed.content, 30)}" z #${this.activitySubject.document.data._id}.`
      );
    } else {
      // W przeciwnym razie zamień treść na nową.
      const oldComment = comments.splice(comments.indexOf(comment), 1, {
        user: comment.user,
        time: comment.time,
        content
      })[0];
      // Tworzenie aktywności dla modyfikacji komentarza.
      this.activityService.logActivity(
        this.activitySubject.id,
        'comments',
        'Zmieniono komentarz',
        `Zmieniono komentarz "${limitWithEllipsis(oldComment.content, 30)}" w #${this.activitySubject.document.data._id}.`
      );
    }
    // Wyemituj zdarzenie, aby uaktualniony komentarz została trwale zapisany poza komponentem.
    this.commentsUpdated.next(comments);
  }

  hasComments() {
    return this.comments && this.comments.length > 0;
  }
}
