import {Directive, Input, Inject, provide, ViewContainerRef, ReflectiveInjector} from '@angular/core';
import {PluginData} from './plugin';
import {PluginService} from './plugin-service';

@Directive({
  selector: 'ngc-plugin-slot'
})
export class PluginSlot {
  @Input() name;

  constructor(@Inject(ViewContainerRef) viewContainerRef, @Inject(PluginService) pluginService) {
    this.viewContainerRef = viewContainerRef;
    this.pluginService = pluginService;
    this.componentRefs = [];
    // Subskrypcja zmian w usłudze plug-inów i ponownia inicjalizacja miejsca, jeśli uległo zmianie.
    this.pluginChangeSubscription = this.pluginService.change.subscribe(() => this.initialize());
  }

  initialize() {
    // Jeśli nie mamy poprawnej nazwy, kończymy działanie.
    if (!this.name) {
      return;
    }

    // Jeśli posiadamy już referencje do komponentów w danym miejscu, usuwamy je i czyścimy listę.
    if (this.componentRefs.length > 0) {
      this.componentRefs.forEach((componentRef) => componentRef.destroy());
      this.componentRefs = [];
    }

    // Dzięki PluginService pobieramy informacje o wszystkich komponentach przewidzanych dla tego miejsca.
    const pluginData = this.pluginService.getPluginData(this.name);
    // Sortujemy komponenty do wstawienia na podstawie priorytetu.
    pluginData.sort(
      (a, b) => a.placement.priority < b.placement.priority ?
        1 : a.placement.priority > b.placement.priority ? -1 : 0);

    // Tworzymy wszystkie komponenty i umieszczamy je we wskazanym miejscu.
    pluginData.forEach((pluginData) => this.instantiatePluginComponent(pluginData));
  }

  // Metoda tworzy pojedynczy komponent i umieszcza go w odpowiednim miejscu.
  instantiatePluginComponent(pluginData) {
    const componentFactory = pluginData.plugin.componentFactories.find(x => x.componentType === pluginData.placement.component);

    // Pobranie obiektu wstrzykiwania dla komponentu nadrzędnego względem położenia miejsca wstawień.
    const contextInjector = this.viewContainerRef.parentInjector;
    // Tworzymy nowy obiekt wstrzykiwania, dodając do niego dostawcę PluginData.
    const childInjector = ReflectiveInjector.resolveAndCreate([{provide: PluginData, useValue: pluginData}], contextInjector);
    // Tworzymy nowy komponent używając kontenera widoku miejsca wstawienia i pobranej fabryki komponentu.
    const componentRef = this.viewContainerRef.createComponent(componentFactory, this.viewContainerRef.length, childInjector);
    this.componentRefs.push(componentRef);
    // Przeprowadź detekcję zmian w sytuacji, gdyby widok nadrzędny stosował techniki OnPush i Detached.
    componentRef.changeDetectorRef.markForCheck();
    componentRef.changeDetectorRef.detectChanges();
  }

  // Jeśli zmieni się nazwa miejsca, ponownie zainicjalizuj wszystkie komponenty.
  ngOnChanges() {
    this.initialize();
  }

  ngOnDestroy() {
    this.pluginChangeSubscription.unsubscribe();
  }
}
