import {Component, ViewEncapsulation, Input, Inject, ElementRef, HostListener, Output, EventEmitter} from '@angular/core';
import template from './activity-slider.html!text';
import styles from './activity-slider.css!text';

@Component({
  selector: 'ngc-activity-slider',
  host: {
    class: 'activity-slider'
  },
  template,
  styles: [styles],
  encapsulation: ViewEncapsulation.Native
})
export class ActivitySlider {
  // Właściwość wejściowa oczekuje listy aktywności.
  @Input() activities;
  // Jeśli w komponencie suwaka zmieni się zakres, wyemitujemy zdarzenie.
  @Output() selectionChange = new EventEmitter();

  constructor(@Inject(ElementRef) elementRef) {
    // Użyjemy elementu hostującego jako bazy wymiarowej przy rysuwaniu SVG.
    this.sliderElement = elementRef.nativeElement;
    // Odstęp po każdej stronie suwaka.
    this.padding = 20;
  }

  ngOnChanges(changes) {
    // Gdy zmieni się parametr wejściowy, musimy przeliczyći ponownie narysować suwak.
    if (changes.activities && changes.activities.currentValue) {
      const activities = changes.activities.currentValue;
      // Na potrzeby późniejszych obliczeń zapamiętujemy czas pierwszej i ostatniej aktywności.
      if (activities.length === 1) {
        // Jeśli to tylko jedna aktywność, oba czasy będą edentyczne.
        this.timeFirst = this.timeLast = activities[0].time;
      } else if (activities.length > 1) {
        // Zapamiętaj pierwszy i ostatni czas.
        this.timeFirst = activities[activities.length - 1].time;
        this.timeLast = activities[0].time;
      } else {
        // Jeśli brak aktywności, użyj aktualny czas jako czas pierwszej i ostatniej aktywności.
        this.timeFirst = this.timeLast = new Date().getTime();
      }

      // Przedział czasu to czas od pierwszej do ostatniej aktywności. Musimy wymusić minimum 1, aby nie mieć problemów w dalszych obliczeniach.
      this.timeSpan = Math.max(1, this.timeLast - this.timeFirst);
      // Przelicz znaczniki wyświetlane na górze suwaka.
      this.computeTicks();
      // Ustaw zaznaczenie na pełny zakres.
      this.selection = {
        start: this.timeFirst,
        end: this.timeLast
      };
      // Wyemituj zdarzenie dla początkowego wyboru.
      this.selectionChange.next(this.selection);
    }
  }

  // Funkcja wylicza położenie pięciu znaczników na suwaku.
  computeTicks() {
    const count = 5;
    const timeSpanTick = this.timeSpan / count;
    this.ticks = Array.from({length: count}).map((element, index) => {
      return this.timeFirst + timeSpanTick * index;
    });
  }

  // Pobranie łącznej szerokości suwaka.
  totalWidth() {
    return this.sliderElement.clientWidth - this.padding * 2;
  }

  // Rzutuje znacznik czasowy na procent położenia.
  projectTime(time) {
    let position = this.padding +
      (time - this.timeFirst) / this.timeSpan * this.totalWidth();
    return position / this.sliderElement.clientWidth * 100;
  }

  // Rzutuje wartość w pikselach na czas. Wymagane do obliczenia znacznika czasowego dla zaznaczenia.
  projectLength(length) {
    return this.timeFirst + (length - this.padding) / this.totalWidth() * this.timeSpan;
  }

  // Jeśli komponent otrzyma zdarzenie mousedown, zaczynamy nowe zaznaczenie.
  @HostListener('mousedown', ['$event'])
  onMouseDown(event) {
    // Rozpoczęcie nowego zaznaczenia poprzez ustawienie czasu początku i końca na aktualne położenie.
    this.selection.start = this.selection.end = this.projectLength(event.offsetX);
    // Zmieniło się zaznaczenie, więc emitujemy zdarzenie.
    this.selectionChange.next(this.selection);
    // Ustawienie znacznika, abyśmy wiedzieli, że odbywa się zaznaczenie.
    this.modifySelection = true;
  }

  // Musimy śledzić ruchy myszy w komponencie suwaka.
  @HostListener('mousemove', ['$event'])
  onMouseMove(event) {
    // Modyfikuj zaznaczenie tylko wtedy, gdy komponent znajduje się w odpowiednim trybie.
    if (this.modifySelection) {
      // Uaktualnij czas zakończenia na podstawie aktualnego położenia kursora myszy.
      this.selection.end = Math.max(this.selection.start, this.projectLength(event.offsetX));
      // Zmieniło się zaznaczenie, więc wyemituj zdarzenie.
      this.selectionChange.next(this.selection);
      // Aby zapobiec efektom ubocznym, zatrzymaj propagację zdarzeń i zapobiegnij domyślnej akcji przeglądarki.
      event.stopPropagation();
      event.preventDefault();
    }
  }

  // Jeśli użytkownik zwolni klawisz myszy, wyjdź z trybu modyfikacji.
  @HostListener('mouseup')
  onMouseUp() {
    this.modifySelection = false;
  }

  // Jeśli użytkownik opuszcza kursorem myszy komponent, wyjdź z trybu modyfikacji.
  @HostListener('mouseleave')
  onMouseLeave() {
    this.modifySelection = false;
  }

  // Podwójne kliknięcie powinno spowodować zaznaczenie całego zakresu.
  @HostListener('dblclick', ['$event'])
  onDoubleClick(event) {
    // Ustawienie pełnego zakresu.
    this.selection = {
      start: this.timeFirst,
      end: this.timeLast
    };
    // Zmieniło się zaznaczenie, więc wyemituj zdarzenie.
    this.selectionChange.next(this.selection);
    // Aby zapobiec efektom ubocznym, zatrzymaj propagację zdarzeń i zapobiegnij domyślnej akcji przeglądarki.
    event.stopPropagation();
    event.preventDefault();
  }

  // Aby zapobiec problemom w niektórych przeglądarkach związanym z trybem zaznaczania, wyłap to zdarzenie i zatrzymaj propagację.
  @HostListener('dragstart', ['$event'])
  onDragStart(event) {
    // Aby zapobiec efektom ubocznym, zatrzymaj propagację zdarzeń i zapobiegnij domyślnej akcji przeglądarki.
    event.stopPropagation();
    event.preventDefault();
  }
}
