// Funkcja służy do określenia wspórzędnych ekranu, w których znajduje się kursor.
function getRangeBoundlingClientRect() {
  if (window.getSelection) {
    const selection = window.getSelection();
    if (!selection.rangeCount) return;

    const range = selection.getRangeAt(0);

    if (!range.collapsed) {
      return range.getBoundingClientRect();
    }

    const dummy = document.createElement('span');
    range.insertNode(dummy);
    const pos = dummy.getBoundingClientRect();
    dummy.parentNode.removeChild(dummy);
    return pos;
  }

  if (document.selection) {
    return document.selection
      .createRange()
      .getBoundingClientRect();
  }
}

// Klasa umożliwia wpisywanie danych wraz z obsługą etykiet.
export class TagInputManager {
  constructor() {
    this.reset();
  }

  reset() {
    // Pole textTag przechowuje całą etykietę, którą wpisał użytkownik (łącznie ze znakiem #).
    this.textTag = '';
    // Pole position zawiera aktualne współrzędne kursora, czyli miejsce, gdzie znajduje się etykieta.
    this.position = null;
  }

  // Zwraca true, jeśli użytkownik wpisuje etykietę.
  hasTextTag() {
    return this.textTag[0] === '#';
  }

  // Wewnętrzna metoda używana, gdy dokonujemy aktualizacji wpisywanej etykiety.
  updateTextTag(textTag, position = this.position) {
    this.textTag = textTag;
    this.position = position;
  }

  // Przy zdarzeniu keydown musimy sprawdzić klawisz cofnięcia i zaktualizować tekst.
  onKeyDown(event) {
    // Jeśli otrzymany kod klawisza dotyczy cofnięcia (kod równy 8), musimy usunąć ostatni znak z etykiety.
    if (event.which === 8 && this.hasTextTag()) {
      this.updateTextTag(this.textTag.slice(0, -1));
    }
  }

  // Obsługa danych wejściowych i realizacja odpowiedniej logiki związanej z tworzenie etykiety.
  onKeyPress(event) {
    const char = String.fromCharCode(event.which);
    if (char === '#') {
      // Jeśli wprowadzony znak to #, możemy rozpocząć generowanie nowej etykiety i określanie aktualnego położenia kursora.
      this.updateTextTag('#', getRangeBoundlingClientRect());
    } else if ((/[\w-]/i).test(this.textTag[0])) {
      // Jeśli wpisany znak nie może być częścią etykiety, resetujemy stan i zakładamy, że anulowano wpisywanie etykiety.
      this.reset();
    } else if (this.hasTextTag()) {
      // JEśli wpisano inny poprawny znak, aktualizujemy tekst etykiety.
      this.updateTextTag(this.textTag + char);
    }
  }
}
