/*
 * Copyright (C) 2009 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gameproject.multiplayerpaddle;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.UUID;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothServerSocket;
import android.bluetooth.BluetoothSocket;
import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.util.Log;

/**
 * Ta klasa wykonuje ca prac zwizan z konfiguracj i zarzdzaniem poczeniami Bluetooth
 * z innymi urzdzeniam. Posiada ona wtek nasuchujcy pocze przychodzcych 
 * wtek poczenia z urzdzeniem i wtek
 * wysyajcy dane po poczeniu
 */
@TargetApi(11)
public class BluetoothChatService {
    // Debugowanie
    private static final String TAG = "BluetoothChatService";
    private static final boolean D = true;

 // Nazwa rekordu SDP uywana podczas tworzenia gniazda serwera
    private static final String NAME = "BluetoothChat";

 // Niepowtarzalny identyfikator UUID tej aplikacji
    private static final UUID MY_UUID = UUID.fromString("fa87c0d0-afac-11de-8a39-0800200c9a66");

 // Pola klasy
    private final BluetoothAdapter mAdapter;
    private final Handler mHandler;
    private AcceptThread mAcceptThread;
    private ConnectThread mConnectThread;
    private ConnectedThread mConnectedThread;
    private int mState;

 // Stae wskazujce biecy stan poczenia
    public static final int STATE_NONE = 0;       // nic nie robimy
    public static final int STATE_LISTEN = 1;     // suchamy nadchodzcych pocze
    public static final int STATE_CONNECTING = 2; // tworzymy poczenie wychodzce
    public static final int STATE_CONNECTED = 3;  // poczony z urzdzeniem zdalnym

    /**
     * Konstruktor. Przygotowuje now sesj Bluetooth.
     * @param context  Kontekst Aktywnoci
     * @param handler  Procedura obsugi wysyajca komunikat do Aktywnoci
     */
    public BluetoothChatService(Context context, Handler handler) {
        mAdapter = BluetoothAdapter.getDefaultAdapter();
        mState = STATE_NONE;
        mHandler = handler;
    }

    /**
     * Ustawia biecy stan poczenia
     * @param state  Warto cakowitoliczbowa okrelajca stan poczenia
     */
    private synchronized void setState(int state) {
        if (D) Log.d(TAG, "setState() " + mState + " -> " + state);
        mState = state;

     // Przeka biec stan do metody obsugi
        mHandler.obtainMessage(BluetoothChat.MESSAGE_STATE_CHANGE, state, -1).sendToTarget();
    }

    /**
     * Zwraca biecy stan poczenia. */
    public synchronized int getState() {
        return mState;
    }

    /**
     * Uruchamia usug komunikacji. W szczeglnoci uruchamia wtek AcceptThread aby rozpocz
     * sesj w trybie nasuchowania. Wywoywana przez metod onResume() klasy Activity*/
    public synchronized void start() {
        if (D) Log.d(TAG, "start");

     // Usu dowolny wtek prbujcy nawiza poczenie 
        if (mConnectThread != null) {mConnectThread.cancel(); mConnectThread = null;}

        // Usu dowolny wtek obsugujcy poczenie
        if (mConnectedThread != null) {mConnectedThread.cancel(); mConnectedThread = null;}

      // Uruchom wtek nasuchujcy na gniedzie Bluetooth 
        if (mAcceptThread == null) {
            mAcceptThread = new AcceptThread();
            mAcceptThread.start();
        }
        setState(STATE_LISTEN);
    }

    /**
     * Uruchom wtek ConnectThread w celu rozpoczecia poczenia ze zdalnym urzdzeniem.
     * @param device  Urzdzenie BluetoothDevice z ktrym naley nawiza poczenie
     */
    public synchronized void connect(BluetoothDevice device) {
        if (D) Log.d(TAG, "pocz z: " + device);

     // Usu dowolny wtek prbujcy nawiza poczenie
        if (mState == STATE_CONNECTING) {
            if (mConnectThread != null) {mConnectThread.cancel(); mConnectThread = null;}
        }

     // Usu dowolny wtek obsugujcy poczenie
        if (mConnectedThread != null) {mConnectedThread.cancel(); mConnectedThread = null;}

     // Uruchom wtek w celu nawizania poczenia z danym urzdzeniem
        mConnectThread = new ConnectThread(device);
        mConnectThread.start();
        setState(STATE_CONNECTING);
    }

    /**
     * Uruchamia wtek ConnectedThread w celu nadzorowania biecego poczenia Bluetooth
     * @param socket  Obiekt BluetoothSocket uyty do nawizania poczenia
     * @param device  Obiekt BluetoothDevice z ktrym urzdzenie jest poczone
     */
    public synchronized void connected(BluetoothSocket socket, BluetoothDevice device) {
        if (D) Log.d(TAG, "poczony");

     // Usu wtek, ktry ukoczy poczenie 
        if (mConnectThread != null) {mConnectThread.cancel(); mConnectThread = null;}

     // Usu dowolny wtek obsugujcy poczenie 
        if (mConnectedThread != null) {mConnectedThread.cancel(); mConnectedThread = null;}

     // Usu wtek AcceptThread, poniewa chcemy poczy si tylko z jednym urzdzeniem 
        if (mAcceptThread != null) {mAcceptThread.cancel(); mAcceptThread = null;}

     // Uruchom wtek w celu zarzdzania poczeniem i transmisj 
        mConnectedThread = new ConnectedThread(socket);
        mConnectedThread.start();

     // Wylij naw poczonego urzdzenia z powrotem do Aktywnoci
        Message msg = mHandler.obtainMessage(BluetoothChat.MESSAGE_DEVICE_NAME);
        Bundle bundle = new Bundle();
        bundle.putString(BluetoothChat.DEVICE_NAME, device.getName());
        msg.setData(bundle);
        mHandler.sendMessage(msg);

        setState(STATE_CONNECTED);
    }

    /**
     * Zatrzymuje wszystkie wtki
     */
    public synchronized void stop() {
        if (D) Log.d(TAG, "stop");
        if (mConnectThread != null) {mConnectThread.cancel(); mConnectThread = null;}
        if (mConnectedThread != null) {mConnectedThread.cancel(); mConnectedThread = null;}
        if (mAcceptThread != null) {mAcceptThread.cancel(); mAcceptThread = null;}
        setState(STATE_NONE);
    }

    /**
     * Zapisuje do wtku ConnectedThread w sposb niesynchronizowany
     * @param out bajty do zapisu
     * @see ConnectedThread#write(byte[])
     */
    public void write(byte[] out) {
    	// Utwrz obiekt tymczasowy 
        ConnectedThread r;
        // Synchronizuj kopi wtku ConnectedThread 
        synchronized (this) {
            if (mState != STATE_CONNECTED) return;
            r = mConnectedThread;
        }
     // Wykonaj niesynchronizowany zapis
        r.write(out);
    }

    /**
     * Wskazuje bd poczenia.
     */
    private void connectionFailed() {
        setState(STATE_LISTEN);

     // Wylij komunikat o bdzie z powrotem do aktywnoci 
        Message msg = mHandler.obtainMessage(BluetoothChat.MESSAGE_TOAST);
        Bundle bundle = new Bundle();
        bundle.putString(BluetoothChat.TOAST, "Bd poczenia z urzdzeniem");
        msg.setData(bundle);
        mHandler.sendMessage(msg);
    }

    /**
     * Wskazuje utrat poczenia.
     */
    private void connectionLost() {
        setState(STATE_LISTEN);

     // Wylij komunikat o bdzie z powrotem do aktywnoci
        Message msg = mHandler.obtainMessage(BluetoothChat.MESSAGE_TOAST);
        Bundle bundle = new Bundle();
        bundle.putString(BluetoothChat.TOAST, "Utracono poczenie z urzdzeniem");
        msg.setData(bundle);
        mHandler.sendMessage(msg);
    }

    /**
     * Ten wtek dziaa w trakcie nasuchiwania nadchodzcych pocze. Dziaa on jako klient po stronie serwea
     * Wtek dziaa do czasu nawizania poczenia (lub jego anulowania)
     */
    private class AcceptThread extends Thread {
    	// Gniazdo serwera lokalnego 
        private final BluetoothServerSocket mmServerSocket;

        public AcceptThread() {
            BluetoothServerSocket tmp = null;

         // Utwrz nowe nasuchujce gniazdo serwera
            try {
                tmp = mAdapter.listenUsingRfcommWithServiceRecord(NAME, MY_UUID);
            } catch (IOException e) {
                Log.e(TAG, "bd funkcji listen()", e);
            }
            mmServerSocket = tmp;
        }

        public void run() {
            if (D) Log.d(TAG, "START mAcceptThread" + this);
            setName("AcceptThread");
            BluetoothSocket socket = null;

         // Dopki brak jest poczenia, nasuchuj na gniedzie sieciowym 
            while (mState != STATE_CONNECTED) {
                try {
                	// To jest wywoanie blokujce, ktre zwrci warto jedynie w przypadku
                	// nawizania poprawnego poczenia lub wystpienia wyjtku 
                    socket = mmServerSocket.accept();
                } catch (IOException e) {
                    Log.e(TAG, "bd funkcji accept()", e);
                    break;
                }

             // Po zaakceptowaniu poczenia
                if (socket != null) {
                    synchronized (BluetoothChatService.this) {
                        switch (mState) {
                        case STATE_LISTEN:
                        case STATE_CONNECTING:
                        	// Sytuacja poprawna. Uruchom wtek poczenia
                            connected(socket, socket.getRemoteDevice());
                            break;
                        case STATE_NONE:
                        case STATE_CONNECTED:
                        	// Poczenie aktywne lub brak gotowoci. Usu nowe gniazdo 
                            try {
                                socket.close();
                            } catch (IOException e) {
                                Log.e(TAG, "Bd zamykania zbdnego gniazda", e);
                            }
                            break;
                        }
                    }
                }
            }
            if (D) Log.i(TAG, "KONIEC mAcceptThread");
        }

        public void cancel() {
            if (D) Log.d(TAG, "anulowanie " + this);
            try {
                mmServerSocket.close();
            } catch (IOException e) {
                Log.e(TAG, "bd funkcji close()", e);
            }
        }
    }


    /**
     * Ten wtek dziaa w trakcie prby nawizania poczenia wychodzcego
     * Dziaa on bardzo prosto - poczenie koczy si sukcesem lub porak
     */
    private class ConnectThread extends Thread {
        private final BluetoothSocket mmSocket;
        private final BluetoothDevice mmDevice;

        public ConnectThread(BluetoothDevice device) {
            mmDevice = device;
            BluetoothSocket tmp = null;

            // Pobierz gniazdo Bluetooth dla poczenia
            // z danym urzdzeniem Bluetooth 
            try {
                tmp = device.createRfcommSocketToServiceRecord(MY_UUID);
            } catch (IOException e) {
                Log.e(TAG, "bd funkcji create()", e);
            }
            mmSocket = tmp;
        }

        public void run() {
            Log.i(TAG, "START mConnectThread");
            setName("ConnectThread");

         // Naley zawsze unika fazy discovery, poniewa zdecydowanie spowalnia ona poczenie 
            mAdapter.cancelDiscovery();

         // Nawi poczenie z gniazdem Bluetooth 
            try {
            	// To jest wywoanie blokujce, ktre zwrci warto jedynie w przypadku
            	// nawizania poprawnego poczenia lub wystpienia wyjtku 
                mmSocket.connect();
            } catch (IOException e) {
                connectionFailed();
             // Zamknij gniazdo 
                try {
                    mmSocket.close();
                } catch (IOException e2) {
                    Log.e(TAG, "bd zamykania gniazda spowodowany bdem poczenia", e2);
                }
             // Ponownie uruchom usug w trybie nasuchu 
                BluetoothChatService.this.start();
                return;
            }

         // Wyzeruj zmienn mConnectThread 
            synchronized (BluetoothChatService.this) {
                mConnectThread = null;
            }

         // Uruchom wtek poczenia 
            connected(mmSocket, mmDevice);
        }

        public void cancel() {
            try {
                mmSocket.close();
            } catch (IOException e) {
                Log.e(TAG, "bd funkcji close()", e);
            }
        }
    }

    /**
     * Ten wtek dziaa podczas poczenia ze zdalnym urzdzeniem.
     * obsuguje on caa transmisj wychodzc oraz przychodzc
     */
    private class ConnectedThread extends Thread {
        private final BluetoothSocket mmSocket;
        private final InputStream mmInStream;
        private final OutputStream mmOutStream;

        public ConnectedThread(BluetoothSocket socket) {
            Log.d(TAG, "utwrz ConnectedThread");
            mmSocket = socket;
            InputStream tmpIn = null;
            OutputStream tmpOut = null;

            // Get the BluetoothSocket input and output streams
            try {
                tmpIn = socket.getInputStream();
                tmpOut = socket.getOutputStream();
            } catch (IOException e) {
                Log.e(TAG, "bd tworzenia gniazd tymczasowyc", e);
            }

            mmInStream = tmpIn;
            mmOutStream = tmpOut;
        }

        public void run() {
            Log.i(TAG, "START mConnectedThread");
            byte[] buffer = new byte[1024];
            int bytes;

         // Dopki jest poczenie, nasuchuj strumie wejciowy
            while (true) {
                try {
                	// Czytaj ze strumienia wejciowego 
                    bytes = mmInStream.read(buffer);

                 // Wylij otrzymane bajty do gwnej aktywnoci
                    mHandler.obtainMessage(BluetoothChat.MESSAGE_READ, bytes, -1, buffer)
                            .sendToTarget();
                } catch (IOException e) {
                    Log.e(TAG, "rozczony", e);
                    connectionLost();
                    break;
                }
            }
        }

        /** 
         * Zapisz do podczonego strumienia OutStream. 
         * @param buffer  Bajty do zapisu 
         */ 
        public void write(byte[] buffer) {
            try {
                mmOutStream.write(buffer);

             // Udostpnij wysany komunikat gwnej aktywnoci
                mHandler.obtainMessage(BluetoothChat.MESSAGE_WRITE, -1, -1, buffer)
                        .sendToTarget();
            } catch (IOException e) {
                Log.e(TAG, "Wyjtek podczas zapisu", e);
            }
        }

        public void cancel() {
            try {
                mmSocket.close();
            } catch (IOException e) {
                Log.e(TAG, "bd funkcji close()", e);
            }
        }
    }
}
