package com.manning.aip.lifecycle;

import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.res.Configuration;
import android.os.Bundle;
import android.util.Log;
import android.widget.RemoteViews;
import android.widget.Toast;

// logcat filterspec
// adb -e logcat "*:s LifecycleExplorer:v"

// see also: adb shell dumpsys activity

/**
 * 
 * Autorzy aplikacji na Androida powinni dobrze rozumie cykl ycia i pokrewiestwo
 * zada w aktywnoci. Niniejsza klasa pomaga zobaczy, jak platforma obsuguje
 * cykl ycia i zadania (uywamy tu komunikatw Toast i rejestrujemy kade wywoanie).  
 * 
 * Cztery stany aktywnoci:
 * ACTIVE/RUNNING (dziaa na pierwszym planie)
 * PAUSED (nie dziaa na pierwszym planie, ale wci widoczna; czciowo 
 * zasania j inna aktywno, niezajmujca caego ekranu)
 * STOPPED (jest cakowicie zasonita przez inn aktywno)
 * KILLED (jest wstrzymana lub zatrzymana; moe zosta zakoczona lub usunita)
 * 
 * Trzeba pamita o trzech ptlach:
 * CAY CZAS YCIA - od onCreate do onDestroy
 * CZAS YCIA, KIEDY AKTYWNO JEST WIDOCZNA - od onStart do onStop
 * CZAS YCIA NA PIERWSZYM PLANIE - od onResume do onPause
 * 
 * Trzy najwaniejsze metody (istniej te inne):
 * onCreate
 * onResume
 * onPause
 * 
 * http://developer.android.com/intl/de/reference/android/app/Activity.html
 * 
 * Uwaga: Uyj instrukcji logcat filterspec, aby zobaczy komunikaty zwizane z debugowaniem:
 * adb -e logcat "*:s LifecycleExplorer:v"
 * 
 * Uwaga: Podaj false w konstruktorze, aby wyczy powiadomienia (ich dua liczba spowalnia aplikacj).
 * 
 * @author ccollins
 *
 */
public abstract class LifecycleActivity extends Activity {

   private static final String LOG_TAG = "LifecycleExplorer";

   private NotificationManager notifyMgr;
   // Przy wartoci domylnej true (lub przekazaniu jej do konstruktora) aplikacja wysya
   // powiadomienia. Jeli jest ich duo, aplikacja moe dziaa powoli. Warto jednak
   // zobaczy, co si w niej dzieje.
   private boolean enableNotifications = true;
   
   private final String className;

   public LifecycleActivity() {
      super();
      this.className = this.getClass().getName();
   }

   public LifecycleActivity(final boolean enableNotifications) {
      this();
      this.enableNotifications = enableNotifications;
   }

   @Override
   public void onCreate(Bundle savedInstanceState) {      
      super.onCreate(savedInstanceState);
      notifyMgr = (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
      debugEvent("onCreate");
   }

   @Override
   protected void onStart() {      
      debugEvent("onStart");
      super.onStart();      
   }

   @Override
   protected void onResume() {      
      debugEvent("onResume");
      super.onResume();      
   }

   @Override
   protected void onPause() {
      debugEvent("onPause");
      super.onPause();
   }

   @Override
   protected void onStop() {
      debugEvent("onStop");
      super.onStop();
   }

   @Override
   protected void onDestroy() {
      debugEvent("onDestroy");
      super.onDestroy();
   }

   //
   // Metody zwizane ze stanem.
   //
   @Override
   protected void onRestoreInstanceState(Bundle savedInstanceState) {      
      debugEvent("onRestoreInstanceState");
      super.onRestoreInstanceState(savedInstanceState);      
   }

   @Override
   protected void onSaveInstanceState(Bundle outState) {      
      debugEvent("onSaveInstanceState");
      super.onSaveInstanceState(outState);      
   }

   //
   // Metod zwizane z konfiguracj. 
   //
   @Override
   public void onConfigurationChanged(Configuration newConfig) {      
      debugEvent("onConfigurationChanged");
      super.onConfigurationChanged(newConfig);      
   }

   @Override
   public Object onRetainNonConfigurationInstance() {
      debugEvent("onRetainNonConfigurationInstance");
      return super.onRetainNonConfigurationInstance();
   }

   //
   // Inne przydatne metody aktywnoci.
   //
   @Override
   public boolean isFinishing() {
      debugEvent("isFinishing");
      return super.isFinishing();
   }

   @Override
   public void finish() {
      super.finish();
   }

   @Override
   public void onLowMemory() {
      Toast.makeText(this, "onLowMemory", Toast.LENGTH_SHORT).show();
      super.onLowMemory();
   }

   //
   // Metoda pomocnicza do wywietlania powiadomie.
   //
   private void debugEvent(final String method) {
      long ts = System.currentTimeMillis();
      Log.d(LOG_TAG, " *** " + method + " " + className + " " + ts);
      if (enableNotifications) {
         Notification notification = new Notification(android.R.drawable.star_big_on, "Zdarzenie cyklu ycia: " + method, 0L);
         RemoteViews notificationContentView = new RemoteViews(getPackageName(), R.layout.custom_notification_layout);
         notification.contentView = notificationContentView;
         notification.contentIntent = PendingIntent.getActivity(this, 0, null, 0);
         notification.flags |= Notification.FLAG_AUTO_CANCEL;
         notificationContentView.setImageViewResource(R.id.image, android.R.drawable.btn_star);
         notificationContentView.setTextViewText(R.id.lifecycle_class, getClass().getName());
         notificationContentView.setTextViewText(R.id.lifecycle_method, method);
         notificationContentView.setTextColor(R.id.lifecycle_method, R.color.black);
         notificationContentView.setTextViewText(R.id.lifecycle_timestamp, Long.toString(ts));
         notifyMgr.notify((int) System.currentTimeMillis(), notification);
      }
   }
}