package com.manning.aip.location;

import android.location.Criteria;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.location.LocationProvider;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.util.Log;

/**
 * Klasa pomocnicza z kodem potrzebnym do ustalania obecnego pooenia przy uyciu
 * dostawcy typu FINE (GPS-a). 
 * <p/>
 * Jeli dostpne jest stosunkowo niedawne pooenie dla dostawcy FINE 
 * (z okresu FIX_RECENT_BUFFER_TIME, ustawionego tu na 30 sekund), aplikacja zwraca je 
 * w komunikacie do jednostki wywoujcej . 
 * <p/>
 * Jeli niedawne pooenie jest niedostpne lub jest zbyt dawne, na okrelony czas
 * uruchamiany jest odbiornik LocationListener. Kiedy odbiornik LocationListener 
 * ustali nowe pooenie lub upyn czas, do jednostki wywoujcej przesyany 
 * jest komunikat.
 * <p/>
 * Przykad ilustrujcy stosowanie klasy w aktywnoci:
 * <p/>
 * <pre>
 *     
 *     Handler handler = new Handler() {
 *        public void handleMessage(Message m) {
 *           Log.d(LOG_TAG, "Zwrcono komunikat: " + m.toString());
 *           if (m.what == LocationHelper.MESSAGE_CODE_LOCATION_FOUND) {
 *              Toast.makeText(Activity.this, "KOMPONENT OBSUGI - wys.:" + m.arg1 + " d.:" + m.arg2, Toast.LENGTH_SHORT)
 *                       .show();
 *           } else if (m.what == LocationHelper.MESSAGE_CODE_LOCATION_NULL) {
 *              Toast.makeText(Activity.this, "KOMPONENT OBSUGI - nie mona ustali pooenia", Toast.LENGTH_SHORT).show();
 *           } else if (m.what == LocationHelper.MESSAGE_CODE_PROVIDER_NOT_PRESENT) {
 *              Toast.makeText(Activity.this, "KOMPONENT OBSUGI - brak dostawcy", Toast.LENGTH_SHORT).show();
 *           }
 *        }
 *     };
 *     
 *     LocationHelper helper = new LocationHelper(locationManager, handler, LOG_TAG);
 *     helper.getCurrentLocation(handler); 
 * </pre> 
 * 
 * @author ccollins
 */
public class LocationHelper {

   public static final int MESSAGE_CODE_LOCATION_FOUND = 1;
   public static final int MESSAGE_CODE_LOCATION_NULL = 2;
   public static final int MESSAGE_CODE_PROVIDER_NOT_PRESENT = 3;   

   private static final int FIX_RECENT_BUFFER_TIME = 30000;

   private LocationManager locationMgr;
   private LocationListener locationListener;
   private Handler handler;
   private Runnable handlerCallback;
   private String providerName;
   private String logTag;
   
   /**
    * W konstruktorze uywany jest obiekt klasy LocationManager i obiekt 
    * klasy Handler do zwracania komunikatw.
    * 
    * @param locationMgr
    * @param handler
    */
   public LocationHelper(LocationManager locationMgr, Handler handler, String logTag) {
      this.locationMgr = locationMgr;
      this.locationListener = new LocationListenerImpl();
      this.handler = handler;      
      this.handlerCallback = new Thread() {
         public void run() {
            endListenForLocation(null);
         }
      };

      Criteria criteria = new Criteria();
      // Klasa Criteria pozwala uzyska dostawc. Mona uy dostawcy typu COARSE,
      // jednak nie dziaa on w emulatorze. Przy ustawieniu FINE uywane s dane z sieci
      // lub GPS w zalenoci od tego, ktry dziaa lepiej w danym miejscu 
      // (w emulatorze trzeba ustawi GPS).
      // UWAGA, okrelanie pooenia za pomoc sieci trzeba wczy: 
      // Settings/Location & Security Settings/Use wireless networks)
      criteria.setAccuracy(Criteria.ACCURACY_FINE);
      this.providerName = locationMgr.getBestProvider(criteria, true);
      
      this.logTag = logTag;
   }

   /**
    * Uruchamia okrelanie pooenia. Obiekt klasy Handler przekazany w 
    * konstruktorze zwraca komunikat z pooeniem.
    * 
    * @param durationSeconds Czas oczekiwania na aktualizacj pooenia
    */
   public void getCurrentLocation(int durationSeconds) {

      if (this.providerName == null) {
         // Zwraca 2/0/0, jeli dostawca jest niedostpny.
         Log.d(logTag, "provideName=null. Dostawca jest wyczony lub nie istnieje.");
         sendLocationToHandler(MESSAGE_CODE_PROVIDER_NOT_PRESENT, 0, 0);
         return;
      }

      // Najpierw naley sprawdzi ostatnie ZNANE pooenie. Jeli jest niedawne, naley  
      // go uy. UWAGA: NIE DZIAA w emulatorze. Jeli za pomoc DDMS-a wylesz
      // rcznie czas lub pooenie, otrzymasz odpowiedni dat, jednak czas ustalenia
      // pooenia zostanie ustawiony na 00:00 i bdzie zwikszany o 1 sekund przy
      // kadym przesaniu nowego pooenia. 
      // Aby przetestowa ten fragment (getLastLocation zwraca wiee dane), 
      // trzeba uy fizycznego urzdzenia.
      Location lastKnown = locationMgr.getLastKnownLocation(providerName);
      if (lastKnown != null && lastKnown.getTime() >= (System.currentTimeMillis() - FIX_RECENT_BUFFER_TIME)) {
         Log.d(logTag, "Ostatnie znane pooenie jest wiee. Mona go uy: " + lastKnown.toString());
         // Zwraca ostatnie znane pooenie w komunikacie od obiektu klasy Handler
         sendLocationToHandler(MESSAGE_CODE_LOCATION_FOUND, (int) (lastKnown.getLatitude() * 1e6),
                  (int) (lastKnown.getLongitude() * 1e6));
      } else {
         // Ostatnie znane pooenie jest stosunkowo dawne lub nie istnieje. Naley 
    	 // uy odbiornika LocationListener i poczeka X sekund na aktualizacj.
         Log.d(logTag, "Ostatnie pooenie NIE jest wiee. Uywanie odbiornika do zaktualizowania pooenia.");
         listenForLocation(providerName, durationSeconds);
      }
   }

   private void sendLocationToHandler(int msgId, int lat, int lon) {
      Message msg = Message.obtain(handler, msgId, lat, lon);
      handler.sendMessage(msg);
   }

   private void listenForLocation(String providerName, int durationSeconds) {
      locationMgr.requestLocationUpdates(providerName, 0, 0, locationListener);
      handler.postDelayed(handlerCallback, durationSeconds * 1000);
   }

   private void endListenForLocation(Location loc) {
      locationMgr.removeUpdates(locationListener);
      handler.removeCallbacks(handlerCallback);
      if (loc != null) {
         sendLocationToHandler(MESSAGE_CODE_LOCATION_FOUND, (int) (loc.getLatitude() * 1e6), (int) (loc.getLongitude() * 1e6));
      } else {
         sendLocationToHandler(MESSAGE_CODE_LOCATION_NULL, 0, 0);
      }
   }

   private class LocationListenerImpl implements LocationListener {
      @Override
      public void onStatusChanged(String provider, int status, Bundle extras) {
         Log.d(logTag, "Zmiana statusu na:" + status);
         switch (status) {
            case LocationProvider.AVAILABLE:
               break;
            case LocationProvider.TEMPORARILY_UNAVAILABLE:
               break;
            case LocationProvider.OUT_OF_SERVICE:
               endListenForLocation(null);
         }
      }

      @Override
      public void onLocationChanged(Location loc) {         
         if (loc == null) {
            return;
         }
         Log.d(logTag, "Zmiana pooenia:" + loc.toString());
         endListenForLocation(loc);
      }

      @Override
      public void onProviderDisabled(String provider) {
         endListenForLocation(null);
      }

      @Override
      public void onProviderEnabled(String provider) {
      }
   }
}