package com.manning.aip.mymoviesdatabase.data;

import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.os.SystemClock;
import android.util.Log;

import com.manning.aip.mymoviesdatabase.Constants;
import com.manning.aip.mymoviesdatabase.data.MovieTable.MovieColumns;
import com.manning.aip.mymoviesdatabase.model.Category;
import com.manning.aip.mymoviesdatabase.model.Movie;

import java.util.List;

/**
 * Klasa DataManagerImpl z operacjami SQL-a i bazodanowymi.
 * Obejmuje klas SQLiteOpenHelper. Uywamy tu obiektw DAO
 * do tworzenia, aktualizowania i usuwania danych oraz 
 * manipulowania nimi na inne sposoby.
 *
 * @author ccollins
 *
 */
public class DataManagerImpl implements DataManager {

   private Context context;

   private SQLiteDatabase db;

   private CategoryDao categoryDao;
   private MovieDao movieDao;
   private MovieCategoryDao movieCategoryDao;

   public DataManagerImpl(Context context) {

      this.context = context;

      SQLiteOpenHelper openHelper = new OpenHelper(this.context);
      db = openHelper.getWritableDatabase();
      Log.i(Constants.LOG_TAG, "Utworzono obiekt klasyDataManagerImpl created. Stan bazy: " + db.isOpen());

      categoryDao = new CategoryDao(db);
      movieDao = new MovieDao(db);
      movieCategoryDao = new MovieCategoryDao(db);
   }

   public SQLiteDatabase getDb() {
      return db;
   }

   private void openDb() {
      if (!db.isOpen()) {
         db = SQLiteDatabase.openDatabase(DataConstants.DATABASE_PATH, null, SQLiteDatabase.OPEN_READWRITE);
         // Poniewa przekazujemy baz do obiektu DAO, trzeba odtworzy obiekt DAO 
         // po ponownym otwarciu bazy.
         categoryDao = new CategoryDao(db);
         movieDao = new MovieDao(db);
         movieCategoryDao = new MovieCategoryDao(db);
      }
   }

   private void closeDb() {
      if (db.isOpen()) {
         db.close();
      }
   }

   private void resetDb() {
      Log.i(Constants.LOG_TAG, "Resetowanie poczenia z baz (zamknicie i ponowne otwarcie).");
      closeDb();
      SystemClock.sleep(500);
      openDb();
   }

   //
   // Nakadki na obiekty DAO (metody do zarzdzania danymi).
   //
   // Pozwalaj ukry, e korzystamy z obiektw DAO. Udostpniane s tylko metody uywane w 
   // aplikacji. Zastosowane rozwizanie pozwala czy obiekty DAO, a logik dziaania 
   // programu umieci w innym miejscu.
   //  

   // Film.
   @Override
   public Movie getMovie(long movieId) {
      Movie movie = movieDao.get(movieId);
      if (movie != null) {
         movie.getCategories().addAll(movieCategoryDao.getCategories(movie.getId()));
      }
      return movie;
   }

   @Override
   public List<Movie> getMovieHeaders() {
      return movieDao.getAll();
   }

   @Override
   public Movie findMovie(String name) {
      Movie movie = movieDao.find(name);
      if (movie != null) {
         movie.getCategories().addAll(movieCategoryDao.getCategories(movie.getId()));
      }
      return movie;
   }

   @Override
   public long saveMovie(Movie movie) {
      // UWAGA: mona te umieci funkcje do manipulowania encjami w klasie DataManagerImpl i  
	  // utworzy "menedera" dla kadej encji. Tu jednak w celu uproszczenia rozwizania
      // bezporednio uywamy obiektw DAO (take przy manipulowaniu encjami).
      long movieId = 0L;

      // Operacje wykonujemy w ramach transakcji (z uwagi na stosowanie kilku tabel).
      try {
         db.beginTransaction();

         // Najpierw zapisujemy film.                                
         movieId = movieDao.save(movie);

         // Nastpnie sprawdzamy, czy kategoria istnieje, i zapisujemy powizanie 
         // filmu z kategori. Wymaga to kilku zapyta, jednak kategorii zwykle jest mao.
         // Mona te zapisa dane i przechwytywa wyjtki, jednak byoby to nieeleganckie.
         if (movie.getCategories().size() > 0) {
            for (Category c : movie.getCategories()) {
               long catId = 0L;
               Category dbCat = categoryDao.find(c.getName());
               if (dbCat == null) {
                  catId = categoryDao.save(c);
               } else {
                  catId = dbCat.getId();
               }
               MovieCategoryKey mcKey = new MovieCategoryKey(movieId, catId);
               if (!movieCategoryDao.exists(mcKey)) {
                  movieCategoryDao.save(mcKey);
               }
            }
         }

         db.setTransactionSuccessful();
      } catch (SQLException e) {
         Log.e(Constants.LOG_TAG, "Bd zapisu filmu (transakcj wycofano)", e);
         movieId = 0L;
      } finally {
         // "Nazwa zastpcza" dla polecenia commit
         db.endTransaction();
      }

      return movieId;
   }

   @Override
   public boolean deleteMovie(long movieId) {
      boolean result = false;
      // UWAGA: zmie kolejno instrukcji, aby zobaczy naruszenie ogranicze 
      // (z uwagi na klucze obce).
      try {
         db.beginTransaction();
         // Uyj metody getMovie zamiast bezporednio movieDao (poniewa trzeba uwzgldni kategorie)
         Movie movie = getMovie(movieId);
         if (movie != null) {
            for (Category c : movie.getCategories()) {
               movieCategoryDao.delete(new MovieCategoryKey(movie.getId(), c.getId()));
            }
            movieDao.delete(movie);
         }
         db.setTransactionSuccessful();
         result = true;
      } catch (SQLException e) {
         Log.e(Constants.LOG_TAG, "Bd przy usuwaniu filmu (transakcj wycofano)", e);
      } finally {
         db.endTransaction();
      }
      return result;
   }

   @Override
   public Cursor getMovieCursor() {
      // Zauwa, e w zapytaniu MUSI wystpowa kolumna _id
      return db.rawQuery("select " + MovieColumns._ID + ", " + MovieColumns.NAME + ", " + MovieColumns.THUMB_URL
               + " from " + MovieTable.TABLE_NAME, null);
   }

   // Kategorie.
   @Override
   public Category getCategory(long categoryId) {
      return categoryDao.get(categoryId);
   }

   @Override
   public List<Category> getAllCategories() {
      return categoryDao.getAll();
   }

   @Override
   public Category findCategory(String name) {
      return categoryDao.find(name);
   }

   @Override
   public long saveCategory(Category category) {
      return categoryDao.save(category);
   }

   @Override
   public void deleteCategory(Category category) {
      categoryDao.delete(category);
   }

}