// DoodleView.java
// Główny obiekt View aplikacji Doodlz.
package com.deitel.doodlz;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Point;
import android.provider.MediaStore;
import android.support.v4.print.PrintHelper;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.widget.Toast;

import java.util.HashMap;
import java.util.Map;

// obiekt View przeznaczony do rysowania
public class DoodleView extends View {
   // zmienna używana do określenia tego, czy użytkownik przesunął palec na odległość pozwalającą na ponowne wykonanie operacji rysowania
   private static final float TOUCH_TOLERANCE = 10;

   private Bitmap bitmap; // obszar rysunku służący do wyświetlania obrazu lub zapisywania go
   private Canvas bitmapCanvas; // do rysowania na bitmapie
   private final Paint paintScreen; // do wyświetlania bitmapy na ekranie
   private final Paint paintLine; // do rysowania linii na bitmapie

   // mapy aktualnie rysowanych ścieżek i punktów na tych ścieżkach
   private final Map<Integer, Path> pathMap = new HashMap<>();
   private final Map<Integer, Point> previousPointMap =  new HashMap<>();

   // Konstruktor inicjalizujący klasę DoodleView
   public DoodleView(Context context, AttributeSet attrs) {
      super(context, attrs); // Konstruktor inicjalizujący klasę DoodleView
      paintScreen = new Paint(); // metoda używana do wyświetlania bitmapy na ekranie

      // określ początkowe parametry rysowanej linii
      paintLine = new Paint();
      paintLine.setAntiAlias(true); // wygładź krawędzie rysowanej linii
      paintLine.setColor(Color.BLACK); // czarny jest kolorem domyślnym
      paintLine.setStyle(Paint.Style.STROKE); // linia ciągła
      paintLine.setStrokeWidth(5); // określ domyślną szerokość linii
      paintLine.setStrokeCap(Paint.Cap.ROUND); // zaokrąglone końce linii
   }

   // tworzy obiekty Bitmap i Canvas na podstawie rozmiaru obiektu View
   @Override
   public void onSizeChanged(int w, int h, int oldW, int oldH) {
      bitmap = Bitmap.createBitmap(getWidth(), getHeight(),
         Bitmap.Config.ARGB_8888);
      bitmapCanvas = new Canvas(bitmap);
      bitmap.eraseColor(Color.WHITE); // skasuj bitmapę (wypełnij ją kolorem białym)
   }

   // wyczyść obraz
   public void clear() {
      pathMap.clear(); // usuń wszystkie ścieżki
      previousPointMap.clear(); // usuń wszystkie wcześniejsze punkty
      bitmap.eraseColor(Color.WHITE); // wyczyść bitmapę
      invalidate(); // odśwież ekran
   }

   //  wybierz kolor malowanej linii
   public void setDrawingColor(int color) {
      paintLine.setColor(color);
   }

   // zwróć kolor malowanej linii
   public int getDrawingColor() {
      return paintLine.getColor();
   }

   // wybierz szerokość malowanej linii
   public void setLineWidth(int width) {
      paintLine.setStrokeWidth(width);
   }

   // zwróć szerokość malowanej linii
   public int getLineWidth() {
      return (int) paintLine.getStrokeWidth();
   }

   // rysuj linie na ekranie, gdy obiekt DoodleView jest odświeżany
   @Override
   protected void onDraw(Canvas canvas) {
      // rysuj ekran tła
      canvas.drawBitmap(bitmap, 0, 0, paintScreen);

      // dla każdej ścieżki, która jest obecnie rysowana
      for (Integer key : pathMap.keySet())
         canvas.drawPath(pathMap.get(key), paintLine); // rysuj linię
   }

   // obsłuż zdarzenie dotyku
   @Override
   public boolean onTouchEvent(MotionEvent event) {
      int action = event.getActionMasked(); // rodzaj zdarzenia
      int actionIndex = event.getActionIndex(); // wskaźnik (tj. palec)

      // Dotyk się rozpoczął czy skończył? A może użytkownik przesuwa palcem po ekranie?
      if (action == MotionEvent.ACTION_DOWN ||
         action == MotionEvent.ACTION_POINTER_DOWN) {
         touchStarted(event.getX(actionIndex), event.getY(actionIndex),
            event.getPointerId(actionIndex));
      }
      else if (action == MotionEvent.ACTION_UP ||
         action == MotionEvent.ACTION_POINTER_UP) {
         touchEnded(event.getPointerId(actionIndex));
      }
      else {
         touchMoved(event);
      }

      invalidate(); // odśwież rysunek
      return true;
   }

   // metoda wywoływana, gdy użytkownik dotknie ekranu
   private void touchStarted(float x, float y, int lineID) {
      Path path; // używana do zapisywania ścieżki dla danego identyfikatora dotknięcia
      Point point; // używana do zapisywania ostatniego punktu ścieżki

      // jeżeli dla danego identyfikatora lineID istnieje już ścieżka
      if (pathMap.containsKey(lineID)) {
         path = pathMap.get(lineID); // uzyskaj ścieżkę
         path.reset(); // resetuje ścieżkę z powodu rozpoczęcia nowego zdarzenia dotyku
         point = previousPointMap.get(lineID); // uzyskaj ostatni punkt ścieżki
      }
      else {
         path = new Path();
         pathMap.put(lineID, path); // dodaj ścieżkę do mapy
         point = new Point(); // utwórz nowy punkt
         previousPointMap.put(lineID, point); // dodaj punkt do mapy
      }

      // przejdź do współrzędnych dotknięcia
      path.moveTo(x, y);
      point.x = (int) x;
      point.y = (int) y;
   }

   // metoda wywoływana, gdy użytkownik przeciągnie palcem po ekranie
   private void touchMoved(MotionEvent event) {
      // dla każdego ze wskaźników danego zdarzenia MotionEven
      for (int i = 0; i < event.getPointerCount(); i++) {
         // uzyskaj identyfikator wskaźnika i jego numer (indeks)
         int pointerID = event.getPointerId(i);
         int pointerIndex = event.findPointerIndex(pointerID);

         // jeżeli występuje ścieżka związana ze wskaźnikiem
         if (pathMap.containsKey(pointerID)) {
            // uzyskaj nowe współrzędne wskaźnika
            float newX = event.getX(pointerIndex);
            float newY = event.getY(pointerIndex);

            // uzyskaj ścieżkę i poprzedni punkt
            // związany z tym wskaźnikiem
            Path path = pathMap.get(pointerID);
            Point point = previousPointMap.get(pointerID);

            // oblicz odległość pokonaną przez wskaźnik od ostatniej aktualizacji jego położenia
            float deltaX = Math.abs(newX - point.x);
            float deltaY = Math.abs(newY - point.y);

            // jeżeli odległość jest wystarczająca
            if (deltaX >= TOUCH_TOLERANCE || deltaY >= TOUCH_TOLERANCE) {
               // przenieś ścieżkę do nowej lokacji
               path.quadTo(point.x, point.y, (newX + point.x) / 2,
                  (newY + point.y) / 2);

               // zapisz nowe współrzędne
               point.x = (int) newX;
               point.y = (int) newY;
            }
         }
      }
   }

   // metoda wywoływana, gdy użytkownik oderwie palec od ekranu
   private void touchEnded(int lineID) {
      Path path = pathMap.get(lineID); // uzyskaj odpowiedni obiekt Path
      bitmapCanvas.drawPath(path, paintLine); // rysuj na bitmapCanvas
      path.reset(); // resetuj obiekt Path
   }

   // zapisz bieżący obraz w galerii
   public void saveImage() {
      // utwórz nazwę pliku dodając do słowa Doodlz bieżący czas
      final String name = "Doodlz" + System.currentTimeMillis() + ".jpg";

      // zapisz obraz w pamięci urządzenia
      String location = MediaStore.Images.Media.insertImage(
         getContext().getContentResolver(), bitmap, name,
         "Doodlz Drawing");

      if (location != null) {
         // wyświetl komunikat informujący o zapisaniu obrazu
         Toast message = Toast.makeText(getContext(),
            R.string.message_saved,
            Toast.LENGTH_SHORT);
         message.setGravity(Gravity.CENTER, message.getXOffset() / 2,
            message.getYOffset() / 2);
         message.show();
      }
      else {
         // wyświetl komunikat informujący o błędzie zapisu
         Toast message = Toast.makeText(getContext(),
            R.string.message_error_saving, Toast.LENGTH_SHORT);
         message.setGravity(Gravity.CENTER, message.getXOffset() / 2,
            message.getYOffset() / 2);
         message.show();
      }
   }

   // drukuj bieżący obraz
   public void printImage() {
      if (PrintHelper.systemSupportsPrint()) {
         // drukuj obraz za pomocą klasy PrintHelper biblioteki Android Support Library
         PrintHelper printHelper = new PrintHelper(getContext());

         // przeskaluj obraz tak, aby zmieścił się w obszarze wydruku, a następnie go wydrukuj
         printHelper.setScaleMode(PrintHelper.SCALE_MODE_FIT);
         printHelper.printBitmap("Doodlz Image", bitmap);
      }
      else {
         // wyświetl komunikat informujący o tym, że system nie pozwala na drukowanie
         Toast message = Toast.makeText(getContext(),
            R.string.message_error_printing, Toast.LENGTH_SHORT);
         message.setGravity(Gravity.CENTER, message.getXOffset() / 2,
            message.getYOffset() / 2);
         message.show();
      }
   }
}

/**************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 **************************************************************************/
