// AddressBookContentProvider.java
// Podklasa klasy ContentProvider służąca do wykonywania operacji na bazie danych.
package com.deitel.addressbook.data;

import android.content.ContentProvider;
import android.content.ContentValues;
import android.content.UriMatcher;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteQueryBuilder;
import android.net.Uri;

import com.deitel.addressbook.R;
import com.deitel.addressbook.data.DatabaseDescription.Contact;
public class AddressBookContentProvider extends ContentProvider {
   // w celu uzyskania dostępu do bazy danych
   private AddressBookDatabaseHelper dbHelper;

   // UriMatcher pomaga obiektowi ContentProvider w określeniu operacji, która ma zostać wykonana
   private static final UriMatcher uriMatcher =
      new UriMatcher(UriMatcher.NO_MATCH);

   // stałe używane przez klasę UriMatcher w celu określenia operacji, która ma zostać wykonana
   private static final int ONE_CONTACT = 1; // wykonaj operację dotyczącą jednego kontaktu
   private static final int CONTACTS = 2; // wykonaj operację dotyczącą tabeli kontaktów

   // blok static konfiguruje obiekt UriMatcher należący do danego obiektu ContentProvider
   static {
      // adres Uri kontaktu o określonym identyfikatorze id (#)
      uriMatcher.addURI(DatabaseDescription.AUTHORITY,
         Contact.TABLE_NAME + "/#", ONE_CONTACT);

      // adres Uri tabeli kontaktów
      uriMatcher.addURI(DatabaseDescription.AUTHORITY,
         Contact.TABLE_NAME, CONTACTS);
   }

   // metoda wywoływana po utworzeniu obiektu AddressBookContentProvider
   @Override
   public boolean onCreate() {
      // utwórz obiekt AddressBookDatabaseHelper
      dbHelper = new AddressBookDatabaseHelper(getContext());
      return true; // operacja tworzenia obiektu ContentProvider została zakończona sukcesem
   }

   // metoda wymagana, nie używamy jej w tej aplikacji, a więc będzie ona zwracała wartość null
   @Override
   public String getType(Uri uri) {
      return null;
   }

   // wygeneruj zapytanie bazy danych
   @Override
   public Cursor query(Uri uri, String[] projection,
      String selection, String[] selectionArgs, String sortOrder) {

      // utwórz obiekt SQLiteQueryBuilder dla zapytania o tabelę kontaktów
      SQLiteQueryBuilder queryBuilder = new SQLiteQueryBuilder();
      queryBuilder.setTables(Contact.TABLE_NAME);

      switch (uriMatcher.match(uri)) {
         case ONE_CONTACT: // wybrany zostanie kontakt o określonym identyfikatorze id
            queryBuilder.appendWhere(
               Contact._ID + "=" + uri.getLastPathSegment());
            break;
         case CONTACTS: // wybrane zostaną wszystkie kontakty
            break;
         default:
            throw new UnsupportedOperationException(
               getContext().getString(R.string.invalid_query_uri) + uri);
      }

      // wykonaj to zapytanie w celu wybrania jednego kontaktu lub wszystkich kontaktów
      Cursor cursor = queryBuilder.query(dbHelper.getReadableDatabase(),
         projection, selection, selectionArgs, null, null, sortOrder);

      // skonfiguruj w celu obserwowania zmian zawartości
      cursor.setNotificationUri(getContext().getContentResolver(), uri);
      return cursor;
   }

   // zapisz nowy kontakt w bazie danych
   @Override
   public Uri insert(Uri uri, ContentValues values) {
      Uri newContactUri = null;

      switch (uriMatcher.match(uri)) {
         case CONTACTS:
            // wstaw nowy kontakt — zakończenie tej operacji sukcesem powoduje zwrot identyfikatora rzędu nowego kontaktu
            long rowId = dbHelper.getWritableDatabase().insert(
               Contact.TABLE_NAME, null, values);

            // utwórz odpowiedni adres Uri w przypadku wstawienia kontaktu do bazy;
            // w przeciwnym wypadku generuj wyjątek
            if (rowId > 0) { // SQLite row IDs start at 1
               newContactUri = Contact.buildContactUri(rowId);

               // powiadom obiekty obserwujące o modyfikacji zawartości bazy danych
               getContext().getContentResolver().notifyChange(uri, null);
            }
            else
               throw new SQLException(
                  getContext().getString(R.string.insert_failed) + uri);
            break;
         default:
            throw new UnsupportedOperationException(
               getContext().getString(R.string.invalid_insert_uri) + uri);
      }

      return newContactUri;
   }

   // zaktualizuj kontakt, który został zapisany wcześniej
   @Override
   public int update(Uri uri, ContentValues values,
      String selection, String[] selectionArgs) {
      int numberOfRowsUpdated; // 1 jeżeli aktualizacja przebiegła pomyślnie; 0 w przeciwnym wypadku

      switch (uriMatcher.match(uri)) {
         case ONE_CONTACT:
            // odczytaj identyfikator kontaktu, który ma zostać zaktualizowany z adresu Uri
            String id = uri.getLastPathSegment();

            // zaktualizuj zawartość
            numberOfRowsUpdated = dbHelper.getWritableDatabase().update(
               Contact.TABLE_NAME, values, Contact._ID + "=" + id,
               selectionArgs);
            break;
         default:
            throw new UnsupportedOperationException(
               getContext().getString(R.string.invalid_update_uri) + uri);
      }

      // jeżeli dokonano modyfikacji bazy danych, to powiadom o tym obiekty obserwujące modyfikacje bazy danych
      if (numberOfRowsUpdated != 0) {
         getContext().getContentResolver().notifyChange(uri, null);
      }

      return numberOfRowsUpdated;
   }

   // skasuj kontakt zapisany wcześniej w bazie danych
   @Override
   public int delete(Uri uri, String selection, String[] selectionArgs) {
      int numberOfRowsDeleted;

      switch (uriMatcher.match(uri)) {
         case ONE_CONTACT:
            // odczytaj identyfikator aktualizowanego kontaktu z adresu uri
            String id = uri.getLastPathSegment();

            // kasuj kontakt
            numberOfRowsDeleted = dbHelper.getWritableDatabase().delete(
               Contact.TABLE_NAME, Contact._ID + "=" + id, selectionArgs);
            break;
         default:
            throw new UnsupportedOperationException(
               getContext().getString(R.string.invalid_delete_uri) + uri);
      }

      // powiadom obiekty obserwujące bazę danych o jej modyfikacji
      if (numberOfRowsDeleted != 0) {
         getContext().getContentResolver().notifyChange(uri, null);
      }

      return numberOfRowsDeleted;
   }
}


/**************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 **************************************************************************/
