// AddEditFragment.java
// Fragment służący do dodawania nowego kontaktu lub edycji zapisanego wcześniej kontaktu.
package com.deitel.addressbook;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.support.design.widget.CoordinatorLayout;
import android.support.design.widget.FloatingActionButton;
import android.support.design.widget.Snackbar;
import android.support.design.widget.TextInputLayout;
import android.support.v4.app.Fragment;
import android.support.v4.app.LoaderManager;
import android.support.v4.content.CursorLoader;
import android.support.v4.content.Loader;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.InputMethodManager;

import com.deitel.addressbook.data.DatabaseDescription.Contact;

public class AddEditFragment extends Fragment
   implements LoaderManager.LoaderCallbacks<Cursor> {

   // definiuje metodę wywołania zwrotnego implementowaną przez aktywność MainActivity
   public interface AddEditFragmentListener {
      // metoda wywoływana, gdy kontakt jest zapisywany
      void onAddEditCompleted(Uri contactUri);
   }

   // stała używana do identyfikacji obiektu Loader
   private static final int CONTACT_LOADER = 0;

   private AddEditFragmentListener listener; // MainActivity
   private Uri contactUri; // adres Uri wybranego kontaktu
   private boolean addingNewContact = true; // dodawanie (true) lub edycja

   // pola EditText, w których wyświetlane mają być dane kontaktu
   private TextInputLayout nameTextInputLayout;
   private TextInputLayout phoneTextInputLayout;
   private TextInputLayout emailTextInputLayout;
   private TextInputLayout streetTextInputLayout;
   private TextInputLayout cityTextInputLayout;
   private TextInputLayout stateTextInputLayout;
   private TextInputLayout zipTextInputLayout;
   private FloatingActionButton saveContactFAB;

   private CoordinatorLayout coordinatorLayout; // obiekt używany wraz z obiektami SnackBar

   // przygotuj AddEditFragmentListener po dołączeniu obiektu Fragment
   @Override
   public void onAttach(Context context) {
      super.onAttach(context);
      listener = (AddEditFragmentListener) context;
   }

   // usuń AddEditFragmentListener po odłączeniu obiektu Fragment
   @Override
   public void onDetach() {
      super.onDetach();
      listener = null;
   }

   // metoda wywoływana, gdy zachodzi potrzeba utworzenia widoku obiektu Fragment
   @Override
   public View onCreateView(
      LayoutInflater inflater, ViewGroup container,
      Bundle savedInstanceState) {
      super.onCreateView(inflater, container, savedInstanceState);
      setHasOptionsMenu(true); // fragment posiada elementy menu, które należy wyświetlić

      // przygotuj do wyświetlenia graficzny interfejs użytkownika i uzyskaj odwołania do wszystkich pól EditText
      View view =
         inflater.inflate(R.layout.fragment_add_edit, container, false);
      nameTextInputLayout =
         (TextInputLayout) view.findViewById(R.id.nameTextInputLayout);
      nameTextInputLayout.getEditText().addTextChangedListener(
         nameChangedListener);
      phoneTextInputLayout =
         (TextInputLayout) view.findViewById(R.id.phoneTextInputLayout);
      emailTextInputLayout =
         (TextInputLayout) view.findViewById(R.id.emailTextInputLayout);
      streetTextInputLayout =
         (TextInputLayout) view.findViewById(R.id.streetTextInputLayout);
      cityTextInputLayout =
         (TextInputLayout) view.findViewById(R.id.cityTextInputLayout);
      stateTextInputLayout =
         (TextInputLayout) view.findViewById(R.id.stateTextInputLayout);
      zipTextInputLayout =
         (TextInputLayout) view.findViewById(R.id.zipTextInputLayout);

      // przygotuj obiekt nasłuchujący zdarzeń przycisku FloatingActionButton
      saveContactFAB = (FloatingActionButton) view.findViewById(
         R.id.saveFloatingActionButton);
      saveContactFAB.setOnClickListener(saveContactButtonClicked);
      updateSaveButtonFAB();

      // kod używany do wyświetlania obiektów SnackBar zawierających krótkie komunikaty
      coordinatorLayout = (CoordinatorLayout) getActivity().findViewById(
         R.id.coordinatorLayout);

      Bundle arguments = getArguments(); // wartość null występuje w przypadku tworzenia nowego kontaktu

      if (arguments != null) {
         addingNewContact = false;
         contactUri = arguments.getParcelable(MainActivity.CONTACT_URI);
      }

      // utwórz obiekt Loader w celu uzyskania kontaktu w przypadku edycji zapisanego wcześniej kontaktu
      if (contactUri != null)
         getLoaderManager().initLoader(CONTACT_LOADER, null, this);

      return view;
   }

   // wykrywa modyfikację tekstu wyświetlanego w polach EditText rozkładu nameTextInputLayout
   // w celu ukrycia lub wyświetlenia przycisku umożliwiającego zapisanie kontaktu
   private final TextWatcher nameChangedListener = new TextWatcher() {
      @Override
      public void beforeTextChanged(CharSequence s, int start, int count,
         int after) {}

      // metoda wywoływana w przypadku modyfikacji tekstu wyświetlanego w rozkładzie nameTextInputLayout
      @Override
      public void onTextChanged(CharSequence s, int start, int before,
         int count) {
         updateSaveButtonFAB();
      }

      @Override
      public void afterTextChanged(Editable s) { }
   };

   // wyświetla przycisk saveButtonFAB dopiero po wprowadzeniu tekstu w polu imienia
   private void updateSaveButtonFAB() {
      String input =
         nameTextInputLayout.getEditText().getText().toString();

      // wyświetl przycisk FloatingActionButton po wprowadzeniu tekstu w polu imienia
      if (input.trim().length() != 0)
         saveContactFAB.show();
      else
         saveContactFAB.hide();
   }

   // reaguje na zdarzenie generowane w momencie zapisywania kontaktu przez użytkownika
   private final View.OnClickListener saveContactButtonClicked =
      new View.OnClickListener() {
         @Override
         public void onClick(View v) {
            // ukryj klawiaturę ekranową
            ((InputMethodManager) getActivity().getSystemService(
               Context.INPUT_METHOD_SERVICE)).hideSoftInputFromWindow(
               getView().getWindowToken(), 0);
            saveContact(); // zapisz kontakt w bazie danych
         }
      };

   // zapisuje dane kontaktu w bazie danych
   private void saveContact() {
      // create ContentValues object containing contact's key-value pairs
      ContentValues contentValues = new ContentValues();
      contentValues.put(Contact.COLUMN_NAME,
         nameTextInputLayout.getEditText().getText().toString());
      contentValues.put(Contact.COLUMN_PHONE,
         phoneTextInputLayout.getEditText().getText().toString());
      contentValues.put(Contact.COLUMN_EMAIL,
         emailTextInputLayout.getEditText().getText().toString());
      contentValues.put(Contact.COLUMN_STREET,
         streetTextInputLayout.getEditText().getText().toString());
      contentValues.put(Contact.COLUMN_CITY,
         cityTextInputLayout.getEditText().getText().toString());
      contentValues.put(Contact.COLUMN_STATE,
         stateTextInputLayout.getEditText().getText().toString());
      contentValues.put(Contact.COLUMN_ZIP,
         zipTextInputLayout.getEditText().getText().toString());

      if (addingNewContact) {
         // zapisuje dane kontaktu w bazie danych
         // wykonania operacji wstawiania kontaktu do bazy za pośrednictwem obiektu AddressBookContentProvider
         Uri newContactUri = getActivity().getContentResolver().insert(
            Contact.CONTENT_URI, contentValues);

         if (newContactUri != null) {
            Snackbar.make(coordinatorLayout,
               R.string.contact_added, Snackbar.LENGTH_LONG).show();
            listener.onAddEditCompleted(newContactUri);
         }
         else {
            Snackbar.make(coordinatorLayout,
               R.string.contact_not_added, Snackbar.LENGTH_LONG).show();
         }
      }
      else {
         // skorzystaj z obiektu ContentResolver w celu
         // wykonania operacji wstawiania kontaktu do bazy za pośrednictwem obiektu AddressBookContentProvider
         int updatedRows = getActivity().getContentResolver().update(
            contactUri, contentValues, null, null);

         if (updatedRows > 0) {
            listener.onAddEditCompleted(contactUri);
            Snackbar.make(coordinatorLayout,
               R.string.contact_updated, Snackbar.LENGTH_LONG).show();
         }
         else {
            Snackbar.make(coordinatorLayout,
               R.string.contact_not_updated, Snackbar.LENGTH_LONG).show();
         }
      }
   }

   // metoda wywoływana przez menedżera LoaderManager w celu utworzenia obiektu Loader
   @Override
   public Loader<Cursor> onCreateLoader(int id, Bundle args) {
      // utwórz obiekt CursorLoader na podstawie argumentu zawierającego identyfikator;
      // w tym fragmencie występuje tylko jeden obiekt Loader, a więc instrukcja switch jest zbędna
      switch (id) {
         case CONTACT_LOADER:
            return new CursorLoader(getActivity(),
               contactUri, // adres Uri kontaktu do wyświetlenia
               null, // rzutowanie wartości null zwraca wszystkie kolumny
               null, // wybranie wartości null zwraca wszystkie rzędy
               null, // brak argumentów selekcji
               null); // kolejność sortowania
         default:
            return null;
      }
   }

   // metoda wywoływana przez menedżera LoaderManager po zakończeniu operacji ładowania danych
   @Override
   public void onLoadFinished(Loader<Cursor> loader, Cursor data) {
      // wyświetl dane kontaktu, jeżeli taki kontakt istnieje w bazie danych
      if (data != null && data.moveToFirst()) {
         // odczytaj indeks kolumny dla każdego elementu danych
         int nameIndex = data.getColumnIndex(Contact.COLUMN_NAME);
         int phoneIndex = data.getColumnIndex(Contact.COLUMN_PHONE);
         int emailIndex = data.getColumnIndex(Contact.COLUMN_EMAIL);
         int streetIndex = data.getColumnIndex(Contact.COLUMN_STREET);
         int cityIndex = data.getColumnIndex(Contact.COLUMN_CITY);
         int stateIndex = data.getColumnIndex(Contact.COLUMN_STATE);
         int zipIndex = data.getColumnIndex(Contact.COLUMN_ZIP);

         // wypełnij pola EditText odczytanymi danymi
         nameTextInputLayout.getEditText().setText(
            data.getString(nameIndex));
         phoneTextInputLayout.getEditText().setText(
            data.getString(phoneIndex));
         emailTextInputLayout.getEditText().setText(
            data.getString(emailIndex));
         streetTextInputLayout.getEditText().setText(
            data.getString(streetIndex));
         cityTextInputLayout.getEditText().setText(
            data.getString(cityIndex));
         stateTextInputLayout.getEditText().setText(
            data.getString(stateIndex));
         zipTextInputLayout.getEditText().setText(
            data.getString(zipIndex));

         updateSaveButtonFAB();
      }
   }

   // metoda wywoływana przez menedżera LoaderManager przy wyzerowywaniu obiektu Loader
   @Override
   public void onLoaderReset(Loader<Cursor> loader) { }
}


/**************************************************************************
 * (C) Copyright 1992-2016 by Deitel & Associates, Inc. and               *
 * Pearson Education, Inc. All Rights Reserved.                           *
 *                                                                        *
 * DISCLAIMER: The authors and publisher of this book have used their     *
 * best efforts in preparing the book. These efforts include the          *
 * development, research, and testing of the theories and programs        *
 * to determine their effectiveness. The authors and publisher make       *
 * no warranty of any kind, expressed or implied, with regard to these    *
 * programs or to the documentation contained in these books. The authors *
 * and publisher shall not be liable in any event for incidental or       *
 * consequential damages in connection with, or arising out of, the       *
 * furnishing, performance, or use of these programs.                     *
 **************************************************************************/
