package com.androidbook.sensor.accel;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.os.Bundle;
import android.os.Environment;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.provider.Settings;
import android.util.Log;

public class MainActivity extends Activity implements SensorEventListener {
    private static final String TAG = "AccelerometerRecordToFile";
    private WakeLock mWakelock = null;
	private SensorManager mMgr;
    private Sensor mAccel;
    private BufferedWriter mLog;
	final private SimpleDateFormat mTimeFormat = new SimpleDateFormat("HH:mm:ss - ");
	private int mSavedTimeout;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);

        mMgr = (SensorManager) this.getSystemService(SENSOR_SERVICE);

        mAccel = mMgr.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);

        // Konfiguruje plik dziennika, w ktrym bd zapisywane informacje. Dodamy go
        // w przypadku, jeli nasza aktywno zostanie uruchomiona ponownie w rodku
        // eksperymentu.
        try {
            String filename = Environment.getExternalStorageDirectory().getAbsolutePath() +
		        "/accel.log";
            mLog = new BufferedWriter(new FileWriter(filename, true));
        }
        catch(Exception e) {
        	Log.e(TAG, "Nie mozna zainicjowac pliku dziennika");
        	e.printStackTrace();
        	finish();
        }

        PowerManager pwrMgr = (PowerManager) this.getSystemService(POWER_SERVICE);
        mWakelock = pwrMgr.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "Accel");
        mWakelock.acquire();

        // Zapisuje biec warto czasu wygaszenia ekranu, a nastpnie
        // zmniejsza j.
        try {
        	mSavedTimeout = Settings.System.getInt(getContentResolver(), 
        	    Settings.System.SCREEN_OFF_TIMEOUT);
        }
        catch(Exception e) {
        	mSavedTimeout = 120000;  // 2 minuty
        }
        Settings.System.putInt(getContentResolver(), 
            Settings.System.SCREEN_OFF_TIMEOUT, 5000);  // 5 sekund
    }

    public BroadcastReceiver mReceiver = new BroadcastReceiver() {
        public void onReceive(Context context, Intent intent) {
            if (Intent.ACTION_SCREEN_OFF.equals(intent.getAction())) {
                writeLog("Ekran zostal wylaczony");
                // Wyrejestruje obiekt nasuchujcy i rejestruje go ponownie.

                mMgr.unregisterListener(MainActivity.this);
                mMgr.registerListener(MainActivity.this, mAccel,
                    SensorManager.SENSOR_DELAY_NORMAL);

            }
        }
    };

    @Override
    protected void onStart() {
    	writeLog("rozpoczynanie...");
        mMgr.registerListener(this, mAccel, SensorManager.SENSOR_DELAY_NORMAL);

        IntentFilter filter = new IntentFilter(Intent.ACTION_SCREEN_OFF);
        registerReceiver(mReceiver, filter);

    	super.onStart();
    }

    @Override
    protected void onStop() {
    	writeLog("zatrzymywanie...");
        mMgr.unregisterListener(this, mAccel);
        unregisterReceiver(mReceiver);
        try {
			mLog.flush();
		} catch (IOException e) {
			// ignoruje wszelkie bdy wystpujce w pliku dziennika
		}
    	super.onStop();
    }

    @Override
    protected void onDestroy() {
    	writeLog("zamykanie...");
    	try {
        	mLog.flush();
        	mLog.close();
    	}
    	catch(Exception e) {
    		// ignoruje wszelkie bdy wystpujce w pliku dziennika
    	}

    	// Odczytuje pierwotn warto czasu wyganicia ekranu
        Settings.System.putInt(getContentResolver(), 
                Settings.System.SCREEN_OFF_TIMEOUT, mSavedTimeout);

        mWakelock.release();

        super.onDestroy();
    }

	public void onAccuracyChanged(Sensor sensor, int accuracy) {
		// ignoruje
	}

	public void onSensorChanged(SensorEvent event) {
		writeLog("Uzyskano zdarzenie czujnika: " + event.values[0] + ", " +
				event.values[1] + ", " + event.values[2]);
	}

	private void writeLog(String str) {
		try {
		    Date now = new Date();
		    mLog.write(mTimeFormat.format(now));
			mLog.write(str);
			mLog.write("\n");
		}
		catch(IOException ioe) {
			ioe.printStackTrace();
		}
	}
}