#!/usr/bin/perl

# Flow_Ford_Fulkerson
#
#       $F = $G->Flow_Ford_Fulkerson($S)
#
#       Zwraca maksimum mozliwosci sieci przeplywu opartej na grafie $G
#       definiowane przez stan $S. Graf $G musi miec argumenty 'capacity' 
#       (zdolnosciprzepustowe) dla krawedzi. $S->{ source } musi zawierac
#       wierzcholek zrodlowy oraz  $S->{ sink } wierzcholek koncowy
#       a $S->{ next_augmenting_path } musi zawierac anonimowa procedure
#       ktora po przeslaniu jej argumentow $F i $S zwroci nastepna
#       potencjalna sciezke rozszerzajaca.
#       Procedura Flow_Ford_Fulkerson zajmie sie rozszerzeniem sciezki.
#       Koncowy graf $F bedzie posiadal atrybuty 'flow' (przeplyw) oraz
#       'capacity' (wolna zdolnosc przepustowa).
#

sub Flow_Ford_Fulkerson {
    my ( $G, $S ) = @_;

    my $F = (ref $G)->new; # Siec przeplywu.
    my @E = $G->edges;
    my ( $u, $v );

    # Kopiujemy krawedzie i zdolnosci przepustowe, zerujemy przeplywy. 
    while (($u, $v) = splice(@E, 0, 2)) {
        $F->add_edge( $u, $v );
        $F->set_attribute( 'capacity', $u, $v,
                           $G->get_attribute( 'capacity', $u, $v ) || 0 );
        $F->set_attribute( 'flow',     $u, $v, 0 );
    }

    # Przechodzimy sciezkami rozszerzajacymi.
    while ( my $ap = $S->{ next_augmenting_path }->( $F, $S ) ) {
        my @aps = @$ap; # segmenty sciezek rozszerzajacych
        my $apr;        # wolna zdolnosc przepustowa sciezki rozszerzajacej
        my $psr;        # wolna zdolnosc przepustowa segmentu sciezki 

        # Znajdujemy minimalna zdolnosc przepustowa sciezki.
        for ( $u = shift @aps; @aps; $u = $v ) {
            $v   = shift @aps;
            $psr = $F->get_attribute( 'capacity', $u, $v ) -
                   $F->get_attribute( 'flow',     $u, $v );
            $apr = $psr
                if $psr >= 0 and ( not defined $apr or $psr < $apr );
        }

        if ( $apr > 0 ) { # "rozszerzamy" sciezke.
            for ( @aps = @$ap, $u = shift @aps; @aps; $u = $v ) {
                $v = shift @aps;
                $F->set_attribute( 'flow',
                                   $u, $v,
                                   $F->get_attribute( 'flow', $u, $v ) +
                                   $apr );
            }
        }
    }

    return $F;
}
