# Flow_Edmonds_Karp
#
#       $F = $G->Flow_Edmonds_Karp($source, $sink)
#
#       Zwraca maksymalny przeplyw w sieci grafu $G korzystajac z
#       algorytmu Edmondsa-Karpa bedacego wariantem Forda-Fulkersona.
#       Wejsciowy graf $G musi posiadac atrybuty 'zdolnosci przepustowej'
#       ('capacity') dla kazdej ze swoich krawedzi; otrzymany graf
#       przeplywow bedzie posiadal atrybuty 'capacity' i 'flow' ('przeplyw')
#       opisujace jego krawedzie.
#
sub Flow_Edmonds_Karp {
    my ( $G, $source, $sink ) = @_;

    my $S;

    $S->{ source } = $source;
    $S->{ sink   } = $sink;
    $S->{ next_augmenting_path } =
        sub {
            my ( $F, $S ) = @_;

            my $source = $S->{ source };
            my $sink   = $S->{ sink   };

            # Inicjujemy nasza sterte "todo" (do zrobienia).
            unless ( exists $S->{ todo } ) {
                # Pierwszy element to tablica asocjacyjna wierzcholkow do tej 
                # pory odwiedzanych, pozostale to sciezka od wierzcholka-zrodla.
                push @{ $S->{ todo } },
                     [ { $source => 1 }, $source ];
            }

            while ( @{ $S->{ todo } } ) {
                # $ap: Kolejna sciezka rozszerzajaca.
                my $ap = shift @{ $S->{ todo } };
                my $sv = shift @$ap;    # Odwiedzane wierzcholki.
                my $v  = $ap->[ -1 ];   # Ostatni wierzcholek sciezki.

                if ( $v eq $sink ) {
                    return $ap;
                } else {
                    foreach my $s ( $G->successors( $v ) ) {
                        unless ( exists $sv->{ $s } ) {
                            push @{ $S->{ todo } },
                                [ { %$sv, $s => 1 }, @$ap, $s ];
                        }
                    }
                }
            }
        };

    return $G->Flow_Ford_Fulkerson( $S );
}

use Graph;

my $roads = Graph->new;

# add_capacity_path() jest definiowana za pomoca 
# procedur add_path() oraz set_attribute('capacity', ...).
$roads->add_capacity_path( qw( LodoweMiasto 20 WaniliowaRownina 18
                               GoraceMiasto ) );
$roads->add_capacity_path( qw( LodoweMiasto 5 TruskawkowePola 7
                               GoraceMiasto ) );
$roads->add_capacity_path( qw( LodoweMiasto 10 CzekoladowyKanion 8
                               OrzechowaGora 10 JezynowyLas 6
                               GoraceMiasto ) );
$roads->add_capacity_path( qw( CzekoladowyKanion 3 TruskawkowePola 0
                               TruskawkowePola ) );
$roads->add_capacity_path( qw( JezynowyLas 15 TruskawkowePola ) );
$roads->add_capacity_path( qw( WaniliowaRownina 11 TruskawkowePola ) );
$roads->add_capacity_path( qw( OrzechowaGora 12 TruskawkowePola ) );

my $f = $roads->Flow_Edmonds_Karp( 'LodoweMiasto', 'GoraceMiasto' );
my @e = $f->edges;

my (@E, @C, @F);
while (my ($u, $v) = splice(@e, 0, 2)) {
    push @E, [ $u, $v ];
    push @C, $f->get_attribute("capacity", $u, $v);
    push @F, $f->get_attribute("flow",     $u, $v);
}

foreach my $e ( map { $_->[0] }
                    sort { $b->[3]      <=> $b->[3] ||
                           $b->[2]      <=> $a->[2] ||
                           $a->[1]->[0] cmp $b->[1]->[0] ||
                           $a->[1]->[1] cmp $b->[1]->[1] }
                        map { [ $_, $E[$_], $C[$_], $F[$_] ] }
                            0..$#E ) {
    printf "%-40s %2d/%2d\n",
           $E[$e]->[0] . "-" . $E[$e]->[1], $F[$e], $C[$e]
}
