import random
import numpy as np

# Uczenie przez wzmacnianie

# Symulator powinien zapewniać przynajmniej wymienione niżej funkcje i informacje:

# -	Inicjowanie środowiska. Ta funkcja wymaga ponownego ustawienia środowiska (w tym agenta) na stan początkowy.

# -	Pobieranie aktualnego stanu środowiska. Ta funkcja powinna zwracać aktualny stan środowiska, 
#   zmieniający się po wykonaniu każdej czynności.

# -	Wykonywanie czynności w środowisku. Ta funkcja powoduje wykonanie przez agenta czynności w środowisku. 
#   Środowisko zmienia się w wyniku czynności, co może skutkować nagrodą.

# - Obliczanie nagród po każdej czynności. Ta funkcja jest powiązana z wykonywaniem czynności w środowisku. 
#   Wymaga obliczenia nagrody za działanie i jego skutków w środowisku.

# - Określanie, czy cel został osiągnięty. Ta funkcja ustala, czy agent osiągnął cel. Ten cel można czasem reprezentować 
#   jako komunikat koniec. W środowisku, w którym celu nie da się osiągnąć, symulator musi sygnalizować zakończenie pracy, gdy 
#   jest to konieczne.

# Ustawianie stałych reprezentujących symbole, nagrody i działania w środowisku.
ROAD_AGENT = '*'

ROAD_EMPTY = ' '
ROAD_EMPTY_REWARD = 100

ROAD_OBSTACLE_CAR = '#'
ROAD_OBSTACLE_CAR_REWARD = -100

ROAD_OBSTACLE_PERSON = '!'
ROAD_OBSTACLE_PERSON_REWARD = -1000

ROAD_GOAL = '@'
ROAD_GOAL_REWARD = 500

ROAD_OUT_OF_BOUNDS = '-'
ROAD_OUT_OF_BOUNDS_REWARD = -5

COMMAND_NORTH = 0
COMMAND_SOUTH = 1
COMMAND_EAST = 2
COMMAND_WEST = 3


# Pobieranie losowych działań.
def get_random_action():
    return random.randint(0, 3)


# Inicjowanie mapy przez określenie punktu początkowego i celu.
DEFAULT_START_X = 0
DEFAULT_START_Y = 0
DEFAULT_GOAL_X = 9
DEFAULT_GOAL_Y = 5
DEFAULT_ROAD_SIZE_X = 10
DEFAULT_ROAD_SIZE_Y = 10
DEFAULT_ROAD = [[' ', '#', '#', '#', ' ', '#', ' ', ' ', '#', ' '],
                [' ', '!', ' ', ' ', ' ', ' ', ' ', '!', ' ', '!'],
                [' ', '#', '#', ' ', '#', '#', ' ', ' ', ' ', '#'],
                [' ', ' ', ' ', ' ', ' ', ' ', '#', '#', ' ', '#'],
                ['#', '!', '!', '#', ' ', '!', ' ', ' ', ' ', ' '],
                [' ', ' ', ' ', '#', ' ', ' ', '#', '#', '#', ' '],
                [' ', '#', '#', '#', '!', ' ', '#', '#', '!', ' '],
                [' ', ' ', ' ', '!', '#', ' ', '!', ' ', ' ', ' '],
                ['#', '!', ' ', ' ', ' ', ' ', ' ', ' ', ' ', '#'],
                [' ', '#', ' ', '#', '#', '@', ' ', '#', ' ', '!']]


# Klasa Simulator zawiera funkcje potrzebne do symulacji działań w środowisku.
class Simulator:

    # Inicjowanie symulatora przez podanie drogi, wielkości, punktu początkowego i punktu docelowego.
    def __init__(self, road, road_size_x, road_size_y, agent_start_x, agent_start_y, goal_x, goal_y):
        self.road_size_x = road_size_x
        self.road_size_y = road_size_y
        self.road = road
        self.rewards = 0
        self.agent_x = agent_start_x
        self.agent_y = agent_start_y
        self.goal_x = goal_x
        self.goal_y = goal_y
        self.states = []

    # Przesunięcie agenta i zwrócenie nagrody na podstawie polecenia. Polecenie określa działanie (ruch na północ, południe, 
    # wschód i zachód).
    def move_agent(self, command):
        reward_update = 0
        next_x = 0
        next_y = 0
        if command == COMMAND_NORTH:
            next_x = self.agent_x - 1
            next_y = self.agent_y
        elif command == COMMAND_SOUTH:
            next_x = self.agent_x + 1
            next_y = self.agent_y
        elif command == COMMAND_EAST:
            next_x = self.agent_x
            next_y = self.agent_y + 1
        elif command == COMMAND_WEST:
            next_x = self.agent_x
            next_y = self.agent_y - 1
        if self.is_within_bounds(next_x, next_y):
            reward_update = self.calculate_movement_reward(next_x, next_y)
            self.agent_x = next_x
            self.agent_y = next_y
            # print('Źródło ', self.player_x, ',', self.player_y)
            # print('Cel ', next_x, ',', next_y)
        else:
            reward_update = ROAD_OUT_OF_BOUNDS_REWARD
        self.rewards += reward_update
        #print('Natychmiastowa nagroda: ', reward_update)
        #print('Ostateczna nagroda: ', self.rewards)
        return reward_update

        # Przesunięcie agenta do punktu docelowego.
    def move(self, target_x, target_y):
        self.agent_x = target_x
        self.agent_y = target_y

    # Obliczanie nagrody za ruch na podstawie działań.
    def calculate_movement_reward(self, next_x, next_y):
        if self.road[next_x][next_y] == ROAD_OBSTACLE_PERSON:
            return ROAD_OBSTACLE_PERSON_REWARD
        elif self.road[next_x][next_y] == ROAD_OBSTACLE_CAR:
            return ROAD_OBSTACLE_CAR_REWARD
        elif self.road[next_x][next_y] == ROAD_GOAL:
            return ROAD_GOAL_REWARD
        else:
            return ROAD_EMPTY_REWARD

    # Określanie, czy docelowy punkt znajduje się w granicach planszy.
    def is_within_bounds(self, target_x, target_y):
        if self.road_size_x > target_x >= 0 and self.road_size_y > target_y >= 0:
            return True
        return False

    # Określanie, czy cel został osiągnięty.
    def is_goal_achieved(self):
        if self.agent_x == self.goal_x and self.agent_y == self.goal_y:
            return True
        return False

    # Pobieranie stanu. Jest to tekst zależny od bezpośrednich sąsiadów bieżącego punktu.
    def get_state(self):
        state = ''
        for x in range(-1, 2):
            for y in range(-1, 2):
                if self.is_within_bounds(x, y):
                    state += self.road[x][y]
                else:
                    state += ROAD_OUT_OF_BOUNDS
        if state not in self.states:
            self.states.append(state)
        return self.states.index(state)

    def print_road(self):
        output = ''
        for x in range(self.road_size_x):
            for y in range(self.road_size_y):
                if x == self.agent_x and y == self.agent_y:
                    output += ROAD_AGENT
                else:
                    output += self.road[x][y]
            output += '\n'
        print('Agent x: ', self.agent_x)
        print('Agent y: ', self.agent_y)
        print(output)


# Wykonywanie znanej ścieżki o najwyższej nagrodzie.
def execute_happy_path():
    simulator = Simulator(DEFAULT_ROAD, DEFAULT_ROAD_SIZE_X, DEFAULT_ROAD_SIZE_Y, DEFAULT_START_X, DEFAULT_START_Y,
                          DEFAULT_GOAL_X, DEFAULT_GOAL_Y)
    simulator.print_road()
    simulator.move_agent(COMMAND_SOUTH)
    simulator.move_agent(COMMAND_SOUTH)
    simulator.move_agent(COMMAND_SOUTH)
    simulator.move_agent(COMMAND_EAST)
    simulator.move_agent(COMMAND_EAST)
    simulator.move_agent(COMMAND_EAST)
    simulator.move_agent(COMMAND_EAST)
    simulator.move_agent(COMMAND_SOUTH)
    simulator.move_agent(COMMAND_SOUTH)
    simulator.move_agent(COMMAND_EAST)
    simulator.move_agent(COMMAND_SOUTH)
    simulator.move_agent(COMMAND_SOUTH)
    simulator.move_agent(COMMAND_SOUTH)
    simulator.move_agent(COMMAND_SOUTH)
    simulator.print_road()


# Wykonywanie znanej ścieżki z karami.
def execute_sad_path():
    simulator = Simulator(DEFAULT_ROAD, DEFAULT_ROAD_SIZE_X, DEFAULT_ROAD_SIZE_Y, DEFAULT_START_X, DEFAULT_START_Y,
                          DEFAULT_GOAL_X, DEFAULT_GOAL_Y)
    simulator.print_road()
    simulator.move_agent(COMMAND_NORTH)
    simulator.move_agent(COMMAND_WEST)
    simulator.move_agent(COMMAND_SOUTH)
    simulator.move_agent(COMMAND_EAST)
    simulator.print_road()


# Stosowanie metody siłowej do czasu dotarcia do celu.
def execute_random_brute_force():
    simulator = Simulator(DEFAULT_ROAD, DEFAULT_ROAD_SIZE_X, DEFAULT_ROAD_SIZE_Y, DEFAULT_START_X, DEFAULT_START_Y,
                          DEFAULT_GOAL_X, DEFAULT_GOAL_Y)
    simulator.print_road()
    while not simulator.is_goal_achieved():
        simulator.move_agent(get_random_action())
        simulator.print_road()
    simulator.print_road()

# Trening tabeli nagród algorytmem Q-learning

# - Inicjowanie symulatora: 
# Ten krok polega na ponownym skonfigurowaniu początkowego stanu środowiska z agentem w neutralnym położeniu.

# - Pobieranie stanu środowiska: 
# Odpowiedzialna za to funkcja powinna zwracać aktualny stan środowiska. Stan zmienia się po wykonaniu każdej akcji.

# - Czy cel został osiągnięty? 
# Należy określić, czy cel został osiągnięty (lub czy symulator uznał eksplorację za zakończoną). W tym przykładzie 
# celem jest dotarcie do właściciela samochodu autonomicznego. Po osiągnięciu tego celu algorytm kończy pracę.

# - Wykonywanie losowej akcji: 
# Należy określić, czy algorytm ma wykonać losową akcję. Jeśli tak, wybierany jest losowy kierunek (północ, południe, 
# wschód lub zachód). Losowe akcje pomagają w eksploracji możliwości w środowisku, aby uniknąć wyuczenia się wąskiego 
# podzbioru czynności.

# - Pobieranie akcji z tabeli nagród: 
# Jeśli algorytm nie wykonuje losowej akcji, sprawdza aktualny stan środowiska w tabeli nagród i wybiera działanie na 
# podstawie wartości z tej tabeli. Więcej o tabeli nagród dowiesz się za chwilę.

# - Wykonywanie akcji w środowisku: 
# Ten krok wymaga wykonania wybranej akcji w środowisku (niezależnie od tego, czy działanie jest losowe, czy zostało 
# pobrane z tabeli nagród). Dana akcja ma skutki w środowisku i powiązana jest z nagrodą.

# - Aktualizowanie tabeli nagród: 
# W dalszym tekście opisane jest aktualizowanie tabeli nagród i kroki wykonywane w tym procesie.


# Współczynnik uczenia jest reprezentowany w książce jako alfa.
# Współczynnik dyskontowania jest reprezentowany jako gamma.
def train_with_q_learning(observation_space, action_space, number_of_iterations, learning_rate, discount,
                          chance_of_random_move):
    # Inicjowanie tabeli nagród.
    q_table = np.zeros([observation_space, action_space], dtype=np.int8)

    # Powtarzanie operacji przez określoną liczbę iteracji.
    for i in range(number_of_iterations):
        # Resetowanie symulatora.
        simulator = Simulator(DEFAULT_ROAD, DEFAULT_ROAD_SIZE_X, DEFAULT_ROAD_SIZE_Y, DEFAULT_START_X, DEFAULT_START_Y,
                              DEFAULT_GOAL_X, DEFAULT_GOAL_Y)
        state = simulator.get_state()
        done = False

        # Kontynuowanie pracy, dopóki symulator nie zakończył pracy.
        while not done:
            action = COMMAND_SOUTH
            # Wybór losowej operacji lub najlepszego ruchu z tabeli nagród dla bieżącego stanu.
            if random.uniform(0, 1) > chance_of_random_move:
                action = get_random_action()
            else:
                action = np.argmax(q_table[state])

            # Wykonywanie wybranej operacji w symulacji; pobieranie zmienionego stanu i nagrody.
            reward = simulator.move_agent(action)
            next_state = simulator.get_state()
            done = simulator.is_goal_achieved()

            print(simulator.get_state())
            print('Stan: ', state)
            print('Akcja: ', action)

            # Obliczanie wartości dla wybranej akcji na podstawie tabeli nagród.
            current_value = q_table[state, action]
            next_state_max_value = np.max(q_table[next_state])
            new_value = (1 - learning_rate) * current_value + learning_rate * (reward + discount * next_state_max_value)
            q_table[state, action] = new_value
            state = next_state
            print(q_table)

    return q_table


# Używanie wyuczonej tabeli nagród do poruszania się po mapie do celu w celu przetestowania procesu uczenia.
def execute_with_q_learning(q_table, number_of_episodes):
    total_epochs = 0
    total_penalties_person = 0
    total_penalties_car = 0

    # Powtarzanie przez określoną liczbę epizodów.
    for episode in range(number_of_episodes):
        # Inicjowanie symulatora.
        simulator = Simulator(DEFAULT_ROAD,
                              DEFAULT_ROAD_SIZE_X, DEFAULT_ROAD_SIZE_Y,
                              DEFAULT_START_X, DEFAULT_START_Y,
                              DEFAULT_GOAL_X, DEFAULT_GOAL_Y)
        state = simulator.get_state()
        epochs = 0
        penalties_person = 0
        penalties_car = 0
        reward = 0
        done = False

        # Wybieranie akcji z tabeli nagród, jeśli symulator nie zakończył pracy.
        while not done:
            action = np.argmax(q_table[state])

            reward = simulator.move_agent(action)
            state = simulator.get_state()
            done = simulator.is_goal_achieved()

            if reward == ROAD_OBSTACLE_PERSON_REWARD:
                penalties_person += 1
            elif reward == ROAD_OBSTACLE_CAR_REWARD:
                penalties_car += 1

            epochs += 1

        # Obliczanie kar.
        total_penalties_person += penalties_person
        total_penalties_car += penalties_car
        total_epochs += epochs

    print('Wyniki po ', number_of_episodes, ' epizodach:')
    print('Średnia liczba kroków na epizod: ', total_epochs / number_of_episodes)
    print('Średnia kara za kolizję z pieszym na epizod: ', total_penalties_person / number_of_episodes)
    print('Średnia kara za kolizję z samochodem na epizod: ', total_penalties_person / number_of_episodes)


if __name__ == '__main__':
    #execute_happy_path()
    #execute_sad_path()
    #execute_random_brute_force()
    LEARNING_RATE = 0.1
    DISCOUNT = 0.6
    CHANCE_OF_RANDOM_ACTION = 0.1
    trained_q_table = train_with_q_learning(4*4*4*4*4*4*4*4, 4, 100, LEARNING_RATE, DISCOUNT, CHANCE_OF_RANDOM_ACTION)
    #execute_q_learning(trained_q_table, 1)
