import random

# Indeksy tablicy z danymi.
# 0 = nazwa, 1 = waga, 2 = wartość, 3 = przystosowanie
KNAPSACK_ITEM_NAME_INDEX = 0
KNAPSACK_ITEM_WEIGHT_INDEX = 1
KNAPSACK_ITEM_VALUE_INDEX = 2

# Mały zbiór danych do problemu plecakowego.
# knapsack_items = [
#    ['Perły', 3, 4],
#    ['Złoto', 7, 7],
#    ['Korona', 4, 5],
#    ['Moneta', 1, 1],
#    ['Topór', 5, 4],
#    ['Miecz', 4, 3],
#    ['Pierścień', 2, 5],
#    ['Puchar', 3, 1],
# ]

# Duży zbiór danych do problemu plecakowego.
knapsack_items = [
    ['Topór', 32252, 68674],
    ['Moneta z brązu', 225790, 471010],
    ['Korona', 468164, 944620],
    ['Diamentowy posążek', 489494, 962094],
    ['Szmaragdowy pas', 35384, 78344],
    ['Skamieliny', 265590, 579152],
    ['Złota moneta', 497911, 902698],
    ['Hełm', 800493, 1686515],
    ['Tusz', 823576, 1688691],
    ['Szkatułka', 552202, 1056157],
    ['Nóż', 323618, 677562],
    ['Długi miecz', 382846, 833132],
    ['Maska', 44676, 99192],
    ['Naszyjnik', 169738, 376418],
    ['Opalowa zawieszka', 610876, 1253986],
    ['Perły', 854190, 1853562],
    ['Kołczan', 671123, 1320297],
    ['Rubinowy pierścień', 698180, 1301637],
    ['Srebrna bransoleta', 446517, 859835],
    ['Czasomierz', 909620, 1677534],
    ['Mundur', 904818, 1910501],
    ['Trucizna', 730061, 1528646],
    ['Wełniany szal', 931932, 1827477],
    ['Kusza', 952360, 2068204],
    ['Stara księga', 926023, 1746556],
    ['Puchar z cynku', 978724, 2100851, 0]
]

# Najlepszy wynik uzyskany za pomocą algorytmu siłowego.
BEST_LARGE_KNAPSACK_SCORE = 13692887

# Algorytmy genetyczne służą do sprawdzania dużej przestrzeni poszukiwań, gdzie szukają dobrego rozwiązania. Należy 
# zauważyć, że algorytm genetyczny nie gwarantuje znalezienia bezwzględnie najlepszego rozwiązania; takie algorytmy 
# próbują znaleźć globalnie najlepsze rozwiązanie i uniknąć ograniczania się do lokalnie najlepszych możliwości.
# Oto ogólny cykl życia algorytmu genetycznego:

# - Tworzenie populacji: tworzenie losowej populacji możliwych rozwiązań.

# - Pomiar przystosowania osobników z populacji: określanie, jak dobre jest dane rozwiązanie.
# Używana jest do tego funkcja przystosowania, która ocenia rozwiązania pod kątem ich jakości.

# - Wybór rodziców na podstawie ich przystosowania: wybieranie określonej liczby par rodziców, które
# wydadzą potomstwo.

# - Reprodukcja osobników na podstawie rodziców: tworzenie potomstwa na podstawie rodziców przez mieszanie
# informacji genetycznej i wprowadzanie drobnych mutacji w potomstwie.

# - Zapełnianie następnego pokolenia: wybieranie z danej populacji osobników i potomstwa, które
# przetrwają do następnego pokolenia.

# Indeksy właściwości osobnika.
INDIVIDUAL_CHROMOSOME_INDEX = 0
INDIVIDUAL_FITNESS_INDEX = 1
INDIVIDUAL_PROBABILITY_INDEX = 2


# Generowanie początkowej populacji losowych osobników.
def generate_initial_population(population_size):
    population = []
    for individual in range(0, population_size):
        individual = ''.join([random.choice('01') for n in range(26)])
        population.append([individual, 0, 0])
    return population


# Obliczanie przystosowania dla każdego osobnika z populacji na podstawie maksymalnej wagi.
def calculate_population_fitness(population, maximum_weight):
    best_fitness = 0
    for individual in population:
        individual_fitness = calculate_individual_fitness(individual[INDIVIDUAL_CHROMOSOME_INDEX], maximum_weight)
        individual[INDIVIDUAL_FITNESS_INDEX] = individual_fitness
        if individual_fitness > best_fitness:
            best_fitness = individual_fitness
        if individual_fitness == -1:
            population.remove(individual)
    return best_fitness


# Obliczanie przystosowania osobnika.
def calculate_individual_fitness(individual, maximum_weight):
    total_individual_weight = 0
    total_individual_value = 0
    for gene_index in range(len(individual)):
        gene_switch = individual[gene_index]
        if gene_switch == '1':
            total_individual_weight += knapsack_items[gene_index][KNAPSACK_ITEM_WEIGHT_INDEX]
            total_individual_value += knapsack_items[gene_index][KNAPSACK_ITEM_VALUE_INDEX]
    if total_individual_weight > maximum_weight:
        return -1
    return total_individual_value


# Ustalanie prawdopodobieństwa wyboru każdego osobnika z populacji.
def set_probabilities(population):
    population_sum = sum(individual[INDIVIDUAL_FITNESS_INDEX] for individual in population)
    for individual in population:
        individual[INDIVIDUAL_PROBABILITY_INDEX] = individual[INDIVIDUAL_FITNESS_INDEX] / population_sum


# Wybór osobników z populacji metodą selekcji ruletkowej.
def roulette_wheel_selection(population, number_of_selections):
    set_probabilities(population)
    slices = []
    total = 0
    for r in range(0, len(population)):
        individual = population[r]
        slices.append([r, total, total + individual[INDIVIDUAL_PROBABILITY_INDEX]])
        total += individual[INDIVIDUAL_PROBABILITY_INDEX]
    chosen_ones = []
    for r in range(number_of_selections):
        spin = random.random()
        result = [s[0] for s in slices if s[1] < spin <= s[2]]
        chosen_ones.append(population[result[0]])
    return chosen_ones


# Generowanie dziecka na podstawie dwóch osobników i jednego punktu krzyżowania.
def one_point_crossover(parent_a, parent_b, xover_point):
    children = [parent_a[:xover_point] + parent_b[xover_point:],
                parent_b[:xover_point] + parent_a[xover_point:]]
    return children


# Generowanie dziecka na podstawie dwóch osobników i dwóch punktów krzyżowania.
def two_point_crossover(parent_a, parent_b, xover_point_1, xover_point_2):
    children = [parent_a[:xover_point_1] + parent_b[xover_point_1:xover_point_2] + parent_a[xover_point_2:],
                parent_b[:xover_point_1] + parent_a[xover_point_1:xover_point_2] + parent_b[xover_point_2:]]
    return children


# Wprowadzanie losowych mutacji u dzieci.
def mutate_children(children, mutation_rate):
    for child in children:
        random_index = random.randint(0, mutation_rate)
        if child[INDIVIDUAL_CHROMOSOME_INDEX][random_index] == '1':
            mutated_child = list(child[INDIVIDUAL_CHROMOSOME_INDEX])
            mutated_child[random_index] = '0'
            child[INDIVIDUAL_CHROMOSOME_INDEX] = mutated_child
        else:
            mutated_child = list(child[INDIVIDUAL_CHROMOSOME_INDEX])
            mutated_child[random_index] = '1'
            child[INDIVIDUAL_CHROMOSOME_INDEX] = mutated_child
    return children


# Reprodukcja dzieci na podstawie wybranych osobników.
def reproduce_children(chosen_selections):
    children = []
    for parent_index in range(len(chosen_selections)//2 - 1):
        children = one_point_crossover(chosen_selections[parent_index],
                                       chosen_selections[parent_index + 1],
                                       CROSSOVER_POSITION_1)
    return children


# Łączenie istniejącej populacji z nowo wygenerowanymi dziećmi.
def merge_population_and_children(population, children):
    return population + children

# Ustawianie parametrów algorytmu genetycznego.
NUMBER_OF_GENERATIONS = 1000
INITIAL_POPULATION_SIZE = 1000
KNAPSACK_WEIGHT_CAPACITY = 6404180
CROSSOVER_POSITION_1 = 13
CROSSOVER_POSITION_2 = 22
MUTATION_RATE = 10
NUMBER_OF_ITERATIONS = 5


# Uruchamianie algorytmu genetycznego.
def run_ga():
    best_global_fitness = 0
    global_population = generate_initial_population(INITIAL_POPULATION_SIZE)
    for generation in range(NUMBER_OF_GENERATIONS):
        current_best_fitness = calculate_population_fitness(global_population, KNAPSACK_WEIGHT_CAPACITY)
        if current_best_fitness > best_global_fitness:
            best_global_fitness = current_best_fitness
        the_chosen = roulette_wheel_selection(global_population, 100)
        the_children = reproduce_children(the_chosen)
        the_children = mutate_children(the_children, MUTATION_RATE)
        global_population = merge_population_and_children(global_population, the_children)
        # print(global_population)

    print('Najlepsze przystosowanie: ', best_global_fitness)
    print('Aktualnie najlepszy wynik: ', BEST_LARGE_KNAPSACK_SCORE)
    print('Trafność: ', best_global_fitness / BEST_LARGE_KNAPSACK_SCORE * 100)
    print('Wielkość populacji: ', len(global_population))

    # calculate_population_fitness(global_population, KNAPSACK_WEIGHT_CAPACITY)
    # the_chosen = roulette_wheel_selection(global_population, 100)
    # the_children = reproduce_children(the_chosen)
    # the_children = mutate_children(the_children)
    # global_population = merge_population_and_children(global_population, the_children)
    # global_population = roulette_wheel_selection(global_population, 100)


# Wykonywanie algorytmu genetycznego przez określoną liczbę iteracji.
for i in range(0, NUMBER_OF_ITERATIONS):
    run_ga()

# print(calculate_individual_fitness('01100100010110001110001001', 6404180))
# print(calculate_individual_fitness('00110101000100011010001000', 6404180))
# print(calculate_individual_fitness('11100100110110000100101101', 6404180))
# print(calculate_individual_fitness('00001000010010101101001001', 6404180))
