import copy
import math


# Ta klasa reprezentuje punkt w labiryncie i zawiera wskaźniki do innych punktów, co pozwala utworzyć ścieżkę.
class Point:
    def __init__(self, x=0, y=0):
        self.x = x
        self.y = y
        self.parent = None
        self.cost = math.inf

    def set_parent(self, p):
        self.parent = p

    def set_cost(self, c):
        self.cost = c

    def print(self):
        print(self.x, ',', self.y)


# Te stałe służą do określania punktów labiryntu znajdujących się w różnych kierunkach względem danego punktu.
NORTH = Point(0, 1)
SOUTH = Point(0, -1)
EAST = Point(1, 0)
WEST = Point(-1, 0)


# Klasa MazePuzzle zawiera mechanizmy gry.
class MazePuzzle:

    WALL = '#'
    EMPTY = '_'
    GOAL = '*'

    # Inicjowanie labiryntu w postaci mapy, która zawiera: * (cel), 0 (puste niezbadane pole) i # (ściana).
    def __init__(self, maze_size_x=5, maze_size_y=5):
        self.maze_size_x = maze_size_x
        self.maze_size_y = maze_size_y
        self.maze = ['*0000',
                     '0###0',
                     '0#0#0',
                     '0#000',
                     '00000']

    def get_current_point_value(self, current_point):
        return self.maze[current_point.x][current_point.y]

    # Zwraca wszystkich poprawnych sąsiadów danego punktu z wyłączeniem punktów spoza labiryntu i ścian. 
    def get_neighbors(self, current_point):
        neighbors = []
        # potential_neighbors = [[0, 1], [0, -1], [1, 0], [-1, 0]]
        potential_neighbors = [[NORTH.x, NORTH.y], [SOUTH.x, SOUTH.y], [EAST.x, EAST.y], [WEST.x, WEST.y]]
        for neighbor in potential_neighbors:
            target_point = Point(current_point.x + neighbor[0], current_point.y + neighbor[1])
            if 0 <= target_point.x < self.maze_size_x and 0 <= target_point.y < self.maze_size_y:
                if self.get_current_point_value(target_point) != '#':
                    neighbors.append(target_point)
        return neighbors

    # Funkcja wizualnie pokazująca zbiór odwiedzonych punktów labiryntu.
    def overlay_points_on_map(self, points):
        overlay_map = copy.deepcopy(self.maze)
        for point in points:
            new_row = overlay_map[point.x][:point.y] + '@' + overlay_map[point.x][point.y + 1:]
            overlay_map[point.x] = new_row

        result = ''
        for x in range(0, self.maze_size_x):
            for y in range(0, self.maze_size_y):
                result += overlay_map[x][y]
            result += '\n'
        print(result)

        return overlay_map

    def print_maze(self):
        result = ''
        for x in range(0, self.maze_size_x):
            for y in range(0, self.maze_size_y):
                result += self.maze[x][y]
            result += '\n'
        print(result)


# Funkcja narzędziowa do pobierania ścieżki jako listy punktów. Przechodzi do rodziców danego węzła do czasu
# natrafienia na korzeń.
def get_path(point):
    path = []
    current_point = point
    while current_point.parent is not None:
        path.append(current_point)
        current_point = current_point.parent
    return path


# Funkcja narzędziowa do wyznaczania długości ścieżki od danego punktu.
def get_path_length(point):
    path = []
    current_point = point
    total_length = 0
    while current_point.parent is not None:
        path.append(current_point)
        total_length += 1
        current_point = current_point.parent
    return total_length


# Funkcja narzędziowa do wyznaczania kosztu ścieżki, jeśli występują dodatkowe koszty ruchów.
def get_path_cost(point):
    path = []
    current_point = point
    total_cost = 0
    while current_point.parent is not None:
        path.append(current_point)
        total_cost += get_cost(get_direction(current_point.parent, current_point))
        current_point = current_point.parent
    return total_cost


# Funkcja narzędziowa do określania kosztu danego ruchu.
def get_move_cost(origin, target):
    return get_cost(get_direction(origin, target))


# Funkcja narzędziowa do określania kierunku ruchu na podstawie punktu źródłowego i docelowego.
def get_direction(origin, target):
    if target.x == origin.x and target.y == origin.y - 1:
        return 'N'
    elif target.x == origin.x and target.y == origin.y + 1:
        return 'S'
    elif target.x == origin.x + 1 and target.y == origin.y:
        return 'E'
    elif target.x == origin.x - 1 and target.y == origin.y:
        return 'W'


# Funkcja narzędziowa do określania kosztu ruchu w danym kierunku. Tu koszt ruchów na północ i południe wynosi 5, a ruchów
# na wschód i zachód jest równy 1.
STANDARD_COST = 1
GRAVITY_COST = 5


def get_cost(direction):
    if direction == 'N' or direction == 'S':
        return GRAVITY_COST
    return STANDARD_COST
