# Drzewa decyzyjne

# Drzewa decyzyjne to struktury opisujące serię decyzji, jakie należy podjąć, by znaleźć rozwiązanie problemu.
# Jeśli zastanawiasz się, czy danego dnia założyć krótkie spodenki, możesz podjąć serię decyzji prowadzącą do 
# wyboru garderoby. Czy będzie zimno w ciągu dnia? Jeśli nie, to czy będziesz wracać wieczorem, kiedy robi się 
# chłodno? Możesz zdecydować się na krótkie spodenki, jeśli dzień jest ciepły i nie będziesz wracać chłodnym wieczorem.
# W procesie budowania drzewa decyzyjnego testowane są wszystkie możliwe pytania, aby ustalić, które z nich najlepiej 
# będzie zadać w określonym punkcie drzewa decyzyjnego. Do sprawdzania pytań używana jest entropia — miara poziomu 
# niepewności w zbiorze danych.

# Dane używane w procesie nauki.
feature_names = ['karaty', 'cena', 'szlif']
feature_examples = [[0.21, 327, 'Akceptowalny'],
                    [0.39, 897, 'Znakomity'],
                    [0.50, 1122, 'Znakomity'],
                    [0.76, 907, 'Akceptowalny'],
                    [0.87, 2757, 'Akceptowalny'],
                    [0.98, 2865, 'Akceptowalny'],
                    [1.13, 3045, 'Znakomity'],
                    [1.34, 3914, 'Znakomity'],
                    [1.67, 4849, 'Znakomity'],
                    [1.81, 5688, 'Znakomity']]


# Klasa Question definiuje cechę i wartość, jaką powinna mieć obserwacja.
class Question:

    def __init__(self, feature, value):
        self.feature = feature
        self.value = value

    def filter(self, example):
        value = example[self.feature]
        return value >= self.value

    def to_string(self):
        return 'Czy ' + feature_names[self.feature] + ' >= ' + str(self.value) + '?'


# Klasa ExamplesNode definiuje węzeł drzewa zawierający sklasyfikowane przykłady.
class ExamplesNode:
    def __init__(self, examples):
        self.examples = find_unique_label_counts(examples)

# Klasa DecisionNode definiuje węzeł drzewa zawierający pytanie i dwa rozgałęzienia.
class DecisionNode:
    def __init__(self, question, branch_true, branch_false):
        self.question = question
        self.branch_true = branch_true
        self.branch_false = branch_false


# Znajduje unikatowe klasy i liczbę wystąpień każdej z nich na podstawie listy obserwacji.
def find_unique_label_counts(examples):
    class_count = {}
    for example in examples:
        label = example[-1]
        if label not in class_count:
            class_count[label] = 0
        class_count[label] += 1
    return class_count


# Dzieli listę obserwacji na podstawie odpowiedzi na pytanie.
def split_examples(examples, question):
    examples_true = []
    examples_false = []
    for example in examples:
        if question.filter(example):
            examples_true.append(example)
        else:
            examples_false.append(example)
    return examples_true, examples_false


# Oblicza indeks Giniego na podstawie listy obserwacji.
def calculate_gini(examples):
    label_counts = find_unique_label_counts(examples)
    uncertainty = 1
    for label in label_counts:
        probability_of_label = label_counts[label] / float(len(examples))
        uncertainty -= probability_of_label ** 2
    return uncertainty


# Oblicza zysk informacyjny na podstawie współczynników Giniego lewego i prawego podzbioru oraz
# aktualnego poziomu niepewności.
def calculate_information_gain(left_gini, right_gini, current_uncertainty):
    total = len(left_gini) + len(right_gini)
    gini_left = calculate_gini(left_gini)
    entropy_left = len(left_gini) / total * gini_left
    gini_right = calculate_gini(right_gini)
    entropy_right = len(right_gini) / total * gini_right
    uncertainty_after = entropy_left + entropy_right
    information_gain = current_uncertainty - uncertainty_after
    return information_gain

# Znajduje najlepszy sposób podziału listy obserwacji na podstawie cech.
def find_best_split(examples, number_of_features):
    best_gain = 0
    best_question = None
    current_uncertainty = calculate_gini(examples)
    for feature_index in range(number_of_features):
        values = set([example[feature_index] for example in examples])
        for value in values:
            question = Question(feature_index, value)
            examples_true, examples_false = split_examples(examples, question)
            if len(examples_true) != 0 or len(examples_false) != 0:
                gain = calculate_information_gain(examples_true, examples_false, current_uncertainty)
                if gain >= best_gain:
                    best_gain, best_question = gain, question
    return best_gain, best_question

# Tworzy drzewo decyzyjne.
def build_tree(examples):
    gain, question = find_best_split(examples, len(examples[0]) - 1)
    if gain == 0:
        return ExamplesNode(examples)
    print('Najlepsze pytanie: ', question.to_string(), '\t', 'Zysk informacyjny: ', "{0:.3f}".format(gain))
    examples_true, examples_false = split_examples(examples, question)
    branch_true = build_tree(examples_true)
    branch_false = build_tree(examples_false)
    return DecisionNode(question, branch_true, branch_false)


def print_tree(node, indentation=''):
    # Obserwacje z bieżącego węzła typu ExamplesNode.
    if isinstance(node, ExamplesNode):
        print(indentation + 'Obserwacje', node.examples)
        return
    # Pytanie z bieżącego węzła typu DecisionNode.
    print(indentation + str(node.question.to_string()))
    # Rekurencyjne znajdowanie obserwacji z odpowiedzią 'True' z bieżącego węzła DecisionNode.
    print(indentation + '---> True:')
    print_tree(node.branch_true, indentation + '\t')
    # Rekurencyjne znajdowanie obserwacji z odpowiedzią 'False' z bieżącego węzła DecisionNode.
    print(indentation + '---> False:')
    print_tree(node.branch_false, indentation + '\t')


tree = build_tree(feature_examples)
print_tree(tree)
