/*****************************************************************************
*                                                                            *
*  ------------------------------- bistree.c ------------------------------  *
*                                                                            *
*****************************************************************************/

#include <stdlib.h>
#include <string.h>

#include "bistree.h"
static void destroy_right(BisTree *tree, BiTreeNode *node);

/*****************************************************************************
*                                                                            *
*  ------------------------------ rotate_left -----------------------------  *
*                                                                            *
*****************************************************************************/

static void rotate_left(BiTreeNode **node) {

BiTreeNode         *left,
                   *grandchild;

left = bitree_left(*node);

if (((AvlNode *)bitree_data(left))->factor == AVL_LFT_HEAVY) {

   /**************************************************************************
   *                                                                         *
   *  Rotacja LL.                                                            *
   *                                                                         *
   **************************************************************************/

   bitree_left(*node) = bitree_right(left);
   bitree_right(left) = *node;
   ((AvlNode *)bitree_data(*node))->factor = AVL_BALANCED;
   ((AvlNode *)bitree_data(left))->factor = AVL_BALANCED;
   *node = left;

   }

else {

   /**************************************************************************
   *                                                                         *
   *  Rotacja LR.                                                            *
   *                                                                         *
   **************************************************************************/

   grandchild = bitree_right(left);
   bitree_right(left) = bitree_left(grandchild);
   bitree_left(grandchild) = left;
   bitree_left(*node) = bitree_right(grandchild);
   bitree_right(grandchild) = *node;

   switch (((AvlNode *)bitree_data(grandchild))->factor) {

      case AVL_LFT_HEAVY:

      ((AvlNode *)bitree_data(*node))->factor = AVL_RGT_HEAVY;
      ((AvlNode *)bitree_data(left))->factor = AVL_BALANCED;
      break;

      case AVL_BALANCED:

      ((AvlNode *)bitree_data(*node))->factor = AVL_BALANCED;
      ((AvlNode *)bitree_data(left))->factor = AVL_BALANCED;
      break;

      case AVL_RGT_HEAVY:

      ((AvlNode *)bitree_data(*node))->factor = AVL_BALANCED;
      ((AvlNode *)bitree_data(left))->factor = AVL_LFT_HEAVY;
      break;

   }

   ((AvlNode *)bitree_data(grandchild))->factor = AVL_BALANCED;
   *node = grandchild;

}

return;

}

/*****************************************************************************
*                                                                            *
*  ----------------------------- rotate_right -----------------------------  *
*                                                                            *
*****************************************************************************/

static void rotate_right(BiTreeNode **node) {

BiTreeNode         *right,
                   *grandchild;

right = bitree_right(*node);

if (((AvlNode *)bitree_data(right))->factor == AVL_RGT_HEAVY) {

   /**************************************************************************
   *                                                                         *
   *  Rotacja RR.                                                            *
   *                                                                         *
   **************************************************************************/

   bitree_right(*node) = bitree_left(right);
   bitree_left(right) = *node;
   ((AvlNode *)bitree_data(*node))->factor = AVL_BALANCED;
   ((AvlNode *)bitree_data(right))->factor = AVL_BALANCED;
   *node = right;

   }

else {

   /**************************************************************************
   *                                                                         *
   *  Rotacja RL.                                                            *
   *                                                                         *
   **************************************************************************/

   grandchild = bitree_left(right);
   bitree_left(right) = bitree_right(grandchild);
   bitree_right(grandchild) = right;
   bitree_right(*node) = bitree_left(grandchild);
   bitree_left(grandchild) = *node;

   switch (((AvlNode *)bitree_data(grandchild))->factor) {

      case AVL_LFT_HEAVY:

      ((AvlNode *)bitree_data(*node))->factor = AVL_BALANCED;
      ((AvlNode *)bitree_data(right))->factor = AVL_RGT_HEAVY;
      break;

      case AVL_BALANCED:

      ((AvlNode *)bitree_data(*node))->factor = AVL_BALANCED;
      ((AvlNode *)bitree_data(right))->factor = AVL_BALANCED;
      break;

      case AVL_RGT_HEAVY:

      ((AvlNode *)bitree_data(*node))->factor = AVL_LFT_HEAVY;
      ((AvlNode *)bitree_data(right))->factor = AVL_BALANCED;
      break;

   }

   ((AvlNode *)bitree_data(grandchild))->factor = AVL_BALANCED;
   *node = grandchild;

}

return;

}

/*****************************************************************************
*                                                                            *
*  ----------------------------- destroy_left -----------------------------  *
*                                                                            *
*****************************************************************************/

static void destroy_left(BisTree *tree, BiTreeNode *node) {

BiTreeNode         **position;

/*****************************************************************************
*                                                                            *
*  Nie mona usuwa pustego drzewa.                                          *
*                                                                            *
*****************************************************************************/

if (bitree_size(tree) == 0)
   return;

/*****************************************************************************
*                                                                            *
*  Sprawdzenie, gdzie naley usuwa wzy.                                   *
*                                                                            *
*****************************************************************************/

if (node == NULL)
   position = &tree->root;
else
   position = &node->left;

/*****************************************************************************
*                                                                            *
*  Usuwanie wzw.                                                          *
*                                                                            *
*****************************************************************************/

if (*position != NULL) {

   destroy_left(tree, *position);
   destroy_right(tree, *position);

   if (tree->destroy != NULL) {

      /***********************************************************************
      *                                                                      *
      *  Wywoanie funkcji uytkownika zwalniajcej pami.                  *
      *                                                                      *
      ***********************************************************************/

      tree->destroy(((AvlNode *)(*position)->data)->data);

   }

   /**************************************************************************
   *                                                                         *
   *  Zwolnienie danych z wza, a potem samego wza.                       *
   *                                                                         *
   **************************************************************************/

   free((*position)->data);
   free(*position);
   *position = NULL;

   /**************************************************************************
   *                                                                         *
   *  Korekta rozmiaru drzewa, aby uwzgldni usunicie wza.               *
   *                                                                         *
   **************************************************************************/

   tree->size--;

}

return;

}

/*****************************************************************************
*                                                                            *
*  ----------------------------- destroy_right ----------------------------  *
*                                                                            *
*****************************************************************************/

static void destroy_right(BisTree *tree, BiTreeNode *node) {

BiTreeNode         **position;

/*****************************************************************************
*                                                                            *
*  Nie mona usuwa pustego drzewa.                                          *
*                                                                            *
*****************************************************************************/

if (bitree_size(tree) == 0)
   return;

/*****************************************************************************
*                                                                            *
*  Sprawdzenie, gdzie usuwa wzy.                                          *
*                                                                            *
*****************************************************************************/

if (node == NULL)
   position = &tree->root;
else
   position = &node->right;

/*****************************************************************************
*                                                                            *
*  Usuwanie wzw.                                                          *
*                                                                            *
*****************************************************************************/

if (*position != NULL) {

   destroy_left(tree, *position);
   destroy_right(tree, *position);

   if (tree->destroy != NULL) {

      /***********************************************************************
      *                                                                      *
      *  Wywoanie fukncji uytkownika zwalniajcej pami.                  *
      *                                                                      *
      ***********************************************************************/

      tree->destroy(((AvlNode *)(*position)->data)->data);

   }

   /**************************************************************************
   *                                                                         *
   *  Zwolnienie danych z wza, a nastpnie samego wza.                   *
   *                                                                         *
   **************************************************************************/

   free((*position)->data);
   free(*position);
   *position = NULL;

   /**************************************************************************
   *                                                                         *
   *  Korekta rozmiaru drzewa, aby uwzgldni usunicie wza.               *
   *                                                                         *
   **************************************************************************/

   tree->size--;

}

return;

}

/*****************************************************************************
*                                                                            *
*  -------------------------------- insert --------------------------------  *
*                                                                            *
*****************************************************************************/

static int insert(BisTree *tree, BiTreeNode **node, const void *data, int
   *balanced) {

AvlNode            *avl_data;

int                cmpval,
                   retval;

/*****************************************************************************
*                                                                            *
*  Wstawianie danych do drzewa.                                              *
*                                                                            *
*****************************************************************************/

if (bitree_is_eob(*node)) {

   /**************************************************************************
   *                                                                         *
   *  Wstawianie wza do pustego drzewa.                                    *
   *                                                                         *
   **************************************************************************/

   if ((avl_data = (AvlNode *)malloc(sizeof(AvlNode))) == NULL)
      return -1;

   avl_data->factor = AVL_BALANCED;
   avl_data->hidden = 0;
   avl_data->data = (void *)data;

   return bitree_ins_left(tree, *node, avl_data);

   }

else {

   /**************************************************************************
   *                                                                         *
   *  Wstawianie do niepustego drzewa.                                       *
   *                                                                         *
   **************************************************************************/

   cmpval = tree->compare(data, ((AvlNode *)bitree_data(*node))->data);

   if (cmpval < 0) {

      /***********************************************************************
      *                                                                      *
      *  Przejcie w lewo.                                                   *
      *                                                                      *
      ***********************************************************************/

      if (bitree_is_eob(bitree_left(*node))) {

         if ((avl_data = (AvlNode *)malloc(sizeof(AvlNode))) == NULL)
            return -1;

         avl_data->factor = AVL_BALANCED;
         avl_data->hidden = 0;
         avl_data->data = (void *)data;

         if (bitree_ins_left(tree, *node, avl_data) != 0)
            return -1;

         *balanced = 0;

         }

      else {

         if ((retval = insert(tree, &bitree_left(*node), data, balanced))
            != 0) {

            return retval;

         }

      }

      /***********************************************************************
      *                                                                      *
      *  Ewentualnie rwnowaenie drzewa.                                    *
      *                                                                      *
      ***********************************************************************/

      if (!(*balanced)) {

         switch (((AvlNode *)bitree_data(*node))->factor) {

            case AVL_LFT_HEAVY:

            rotate_left(node);
            *balanced = 1;
            break;

            case AVL_BALANCED:

            ((AvlNode *)bitree_data(*node))->factor = AVL_LFT_HEAVY;
            break;

            case AVL_RGT_HEAVY:

            ((AvlNode *)bitree_data(*node))->factor = AVL_BALANCED;
            *balanced = 1;

         }

      }

      } /* if (cmpval < 0) */

   else if (cmpval > 0) {

      /***********************************************************************
      *                                                                      *
      *  Przejcie w prawo.                                                  *
      *                                                                      *
      ***********************************************************************/

      if (bitree_is_eob(bitree_right(*node))) {

         if ((avl_data = (AvlNode *)malloc(sizeof(AvlNode))) == NULL)
            return -1;

         avl_data->factor = AVL_BALANCED;
         avl_data->hidden = 0;
         avl_data->data = (void *)data;

         if (bitree_ins_right(tree, *node, avl_data) != 0)
            return -1;

         *balanced = 0;

         }

      else {

         if ((retval = insert(tree, &bitree_right(*node), data, balanced))
            != 0) {

            return retval;

         }

      }

      /***********************************************************************
      *                                                                      *
      *  Ewentualnie rwnowaenie drzewa.                                    *
      *                                                                      *
      ***********************************************************************/

      if (!(*balanced)) {

         switch (((AvlNode *)bitree_data(*node))->factor) {

            case AVL_LFT_HEAVY:

            ((AvlNode *)bitree_data(*node))->factor = AVL_BALANCED;
            *balanced = 1;
            break;

            case AVL_BALANCED:

            ((AvlNode *)bitree_data(*node))->factor = AVL_RGT_HEAVY;
            break;

            case AVL_RGT_HEAVY:

            rotate_right(node);
            *balanced = 1;

         }

      }

      } /* if (cmpval > 0) */

   else {

      /***********************************************************************
      *                                                                      *
      *  Odnajdowanie kopii danych.                                          *
      *                                                                      *
      ***********************************************************************/

      if (!((AvlNode *)bitree_data(*node))->hidden) {

         /********************************************************************
         *                                                                   *
         *  Nic nie robimy, bo dane s w drzewie i nie s ukryte.            *
         *                                                                   *
         ********************************************************************/

         return 1;

         }

      else {

         /********************************************************************
         *                                                                   *
         *  Wstawienie nowych danych i oznaczenie ich jako nieukrytych.      *
         *                                                                   *
         ********************************************************************/

         if (tree->destroy != NULL) {

            /*****************************************************************
            *                                                                *
            *  Usunicie ukrytych danych, gdy zostan podmienione.          *
            *                                                                *
            *****************************************************************/

            tree->destroy(((AvlNode *)bitree_data(*node))->data);

         }

         ((AvlNode *)bitree_data(*node))->data = (void *)data;
         ((AvlNode *)bitree_data(*node))->hidden = 0;

         /********************************************************************
         *                                                                   *
         *  Rwnowaenie zbdne, gdy struktura drzewa nie ulega zmianie.   *
         *                                                                   *
         ********************************************************************/

         *balanced = 1;

      }

   }

}

return 0;

}

/*****************************************************************************
*                                                                            *
*  --------------------------------- hide ---------------------------------  *
*                                                                            *
*****************************************************************************/

static int hide(BisTree *tree, BiTreeNode *node, const void *data) {

int                cmpval,
                   retval;

if (bitree_is_eob(node)) {

   /**************************************************************************
   *                                                                         *
   *  Zwracamy informacj, e danych nie znaleziono.                         *
   *                                                                         *
   **************************************************************************/

   return -1;

}

cmpval = tree->compare(data, ((AvlNode *)bitree_data(node))->data);

if (cmpval < 0) {

   /**************************************************************************
   *                                                                         *
   *  Przejcie w lewo.                                                      *
   *                                                                         *
   **************************************************************************/

   retval = hide(tree, bitree_left(node), data);

   }

else if (cmpval > 0) {

   /**************************************************************************
   *                                                                         *
   *  Przejcie w prawo.                                                     *
   *                                                                         *
   **************************************************************************/

   retval = hide(tree, bitree_right(node), data);

   }

else {

   /**************************************************************************
   *                                                                         *
   *  Oznaczenie wza jako ukrytego.                                        *
   *                                                                         *
   **************************************************************************/

   ((AvlNode *)bitree_data(node))->hidden = 1;
   retval = 0;

}

return retval;

}

/*****************************************************************************
*                                                                            *
*  -------------------------------- lookup --------------------------------  *
*                                                                            *
*****************************************************************************/

static int lookup(BisTree *tree, BiTreeNode *node, void **data) {

int                cmpval,
                   retval;

if (bitree_is_eob(node)) {

   /**************************************************************************
   *                                                                         *
   *  Zwracamy informacj, e wza nie znaleziono.                          *
   *                                                                         *
   **************************************************************************/

   return -1;

}

cmpval = tree->compare(*data, ((AvlNode *)bitree_data(node))->data);

if (cmpval < 0) {

   /**************************************************************************
   *                                                                         *
   *  Przejcie w lewo.                                                      *
   *                                                                         *
   **************************************************************************/

   retval = lookup(tree, bitree_left(node), data);

   }

else if (cmpval > 0) {

   /**************************************************************************
   *                                                                         *
   *  Przejcie w prawo.                                                     *
   *                                                                         *
   **************************************************************************/

   retval = lookup(tree, bitree_right(node), data);

   }

else {

   if (!((AvlNode *)bitree_data(node))->hidden) {

      /***********************************************************************
      *                                                                      *
      *  Przekazanie danych z powrotem do drzewa.                            *
      *                                                                      *
      ***********************************************************************/

      *data = ((AvlNode *)bitree_data(node))->data;
      retval = 0;

      }

   else {

      /***********************************************************************
      *                                                                      *
      *  Zwracamy informacj, e danych nie znaleziono.                      *
      *                                                                      *
      ***********************************************************************/

      return -1;

   }

}

return retval;

}

/*****************************************************************************
*                                                                            *
*  ----------------------------- bistree_init -----------------------------  *
*                                                                            *
*****************************************************************************/

void bistree_init(BisTree *tree, int (*compare)(const void *key1, const void
   *key2), void (*destroy)(void *data)) {

/*****************************************************************************
*                                                                            *
*  Inicjalizacja drzewa.                                                     *
*                                                                            *
*****************************************************************************/

bitree_init(tree, destroy);
tree->compare = compare;

return;

}

/*****************************************************************************
*                                                                            *
*  ---------------------------- bistree_destroy ---------------------------  *
*                                                                            *
*****************************************************************************/

void bistree_destroy(BisTree *tree) {

/*****************************************************************************
*                                                                            *
*  Usuwanie z drzewa wszystkich wzw.                                      *
*                                                                            *
*****************************************************************************/

destroy_left(tree, NULL);

/*****************************************************************************
*                                                                            *
*  Nie mona ju na drzewie robi adnych operacji, ale na wszelki wypadek   *
*  czycimy struktur.                                                       *
*                                                                            *
*****************************************************************************/

memset(tree, 0, sizeof(BisTree));

return;

}

/*****************************************************************************
*                                                                            *
*  ---------------------------- bistree_insert ----------------------------  *
*                                                                            *
*****************************************************************************/

int bistree_insert(BisTree *tree, const void *data) {

int                balanced = 0;

return insert(tree, &bitree_root(tree), data, &balanced);

}

/*****************************************************************************
*                                                                            *
*  ---------------------------- bistree_remove ----------------------------  *
*                                                                            *
*****************************************************************************/

int bistree_remove(BisTree *tree, const void *data) {

return hide(tree, bitree_root(tree), data);

}

/*****************************************************************************
*                                                                            *
*  ---------------------------- bistree_lookup ----------------------------  *
*                                                                            *
*****************************************************************************/

int bistree_lookup(BisTree *tree, void **data) {

return lookup(tree, bitree_root(tree), data);

}
