/*****************************************************************************
*                                                                            *
*  -------------------------------- heap.c --------------------------------  *
*                                                                            *
*****************************************************************************/

#include <stdlib.h>
#include <string.h>

#include "heap.h"

/*****************************************************************************
*                                                                            *
*  Lokalne makrodefinicje uywane w implementacji sterty.                    *
*                                                                            *
*****************************************************************************/

#define heap_parent(npos) ((int)(((npos) - 1) / 2))

#define heap_left(npos) (((npos) * 2) + 1)

#define heap_right(npos) (((npos) * 2) + 2)

/*****************************************************************************
*                                                                            *
*  ------------------------------- heap_init ------------------------------  *
*                                                                            *
*****************************************************************************/

void heap_init(Heap *heap, int (*compare)(const void *key1, const void *key2),
   void (*destroy)(void *data)) {

/*****************************************************************************
*                                                                            *
*  Inicjalizacja sterty.                                                     *
*                                                                            *
*****************************************************************************/

heap->size = 0;
heap->compare = compare;
heap->destroy = destroy;
heap->tree = NULL;

return;

}

/*****************************************************************************
*                                                                            *
*  ----------------------------- heap_destroy -----------------------------  *
*                                                                            *
*****************************************************************************/

void heap_destroy(Heap *heap) {

int                i;

/*****************************************************************************
*                                                                            *
*  Usunicie ze sterty wszystkich wzw.                                    *
*                                                                            *
*****************************************************************************/

if (heap->destroy != NULL) {

   for (i = 0; i < heap_size(heap); i++) {

      /***********************************************************************
      *                                                                      *
      *  Wywoanie funkcji uytkownika zwalniajcej zaalokowan pami.      *
      *                                                                      *
      ***********************************************************************/

      heap->destroy(heap->tree[i]);

   }

}

/*****************************************************************************
*                                                                            *
*  Zwolnienie pami zaalokowanaj na sam stert.                            *
*                                                                            *
*****************************************************************************/

free(heap->tree);

/*****************************************************************************
*                                                                            *
*  Niedozwolone s ju adne operacje, ale na wszelki wypadek czycimy       *
*  struktur danych sterty.                                                  *
*                                                                            *
*****************************************************************************/

memset(heap, 0, sizeof(Heap));

return;

}

/*****************************************************************************
*                                                                            *
*  ------------------------------ heap_insert -----------------------------  *
*                                                                            *
*****************************************************************************/

int heap_insert(Heap *heap, const void *data) {

void               *temp;

int                ipos,
                   ppos;

/*****************************************************************************
*                                                                            *
*  Alokacja pamici na wze.                                                *
*                                                                            *
*****************************************************************************/

if ((temp = (void **)realloc(heap->tree, (heap_size(heap) + 1) * sizeof
   (void *))) == NULL) {

   return -1;

   }

else {

   heap->tree = temp;

}

/*****************************************************************************
*                                                                            *
*  Wstawienie wza na koniec.                                               *
*                                                                            *
*****************************************************************************/

heap->tree[heap_size(heap)] = (void *)data;

/*****************************************************************************
*                                                                            *
*  Ponowne przeksztacenie drzewa w stert przez przesunicie nowego wza   *
*  w gr.                                                                   *
*                                                                            *
*****************************************************************************/

ipos = heap_size(heap);
ppos = heap_parent(ipos);

while (ipos > 0 && heap->compare(heap->tree[ppos], heap->tree[ipos]) < 0) {

   /**************************************************************************
   *                                                                         *
   *  Zamiana wartoci wza biecego z jego rodzicem.                      *
   *                                                                         *
   **************************************************************************/

   temp = heap->tree[ppos];
   heap->tree[ppos] = heap->tree[ipos];
   heap->tree[ipos] = temp;

   /**************************************************************************
   *                                                                         *
   *  Przesunicie wza w gr drzewa.                                      *
   *                                                                         *
   **************************************************************************/

   ipos = ppos;
   ppos = heap_parent(ipos);

}

/*****************************************************************************
*                                                                            *
*  Korekta rozmiaru sterty, aby uwzgldni wstawienie wza.                 *
*                                                                            *
*****************************************************************************/

heap->size++;

return 0;

}

/*****************************************************************************
*                                                                            *
*  ----------------------------- heap_extract -----------------------------  *
*                                                                            *
*****************************************************************************/

int heap_extract(Heap *heap, void **data) {

void               *save,
                   *temp;

int                ipos,
                   lpos,
                   rpos,
                   mpos;

/*****************************************************************************
*                                                                            *
*  Nie mona pobiera wza z pustej sterty.                                 *
*                                                                            *
*****************************************************************************/

if (heap_size(heap) == 0)
   return -1;

/*****************************************************************************
*                                                                            *
*  Pobranie wza ze szczytu sterty.                                         *
*                                                                            *
*****************************************************************************/

*data = heap->tree[0];

/*****************************************************************************
*                                                                            *
*  Korekta iloci pamici zajmowanej przez stert.                           *
*                                                                            *
*****************************************************************************/

save = heap->tree[heap_size(heap) - 1];

if (heap_size(heap) - 1 > 0) {

   if ((temp = (void **)realloc(heap->tree, (heap_size(heap) - 1) * sizeof
      (void *))) == NULL) {

      return -1;

      }

   else {

      heap->tree = temp;

   }

   /**************************************************************************
   *                                                                         *
   *  Korekta rozmiaru sterty, aby uwzgldni pobranie wza.                *
   *                                                                         *
   **************************************************************************/

   heap->size--;

   }

else {

   /**************************************************************************
   *                                                                         *
   *  Obsuga sterty po pobraniu ostatniego wza.                           *
   *                                                                         *
   **************************************************************************/

   free(heap->tree);
   heap->tree = NULL;
   heap->size = 0;
   return 0;

}

/*****************************************************************************
*                                                                            *
*  Kopiowanie ostatniego wza do korzenia.                                  *
*                                                                            *
*****************************************************************************/

heap->tree[0] = save;

/*****************************************************************************
*                                                                            *
*  Ponowne przeksztacenie drzewa w stert przez przesuwanie nowego korzenia *
*  w d drzewa.                                                             *
*                                                                            *
*****************************************************************************/

ipos = 0;
lpos = heap_left(ipos);
rpos = heap_right(ipos);

while (1) {

   /**************************************************************************
   *                                                                         *
   *  Wybr dziecka, ktre zostanie zamienione z wzem biecym.            *
   *                                                                         *
   **************************************************************************/

   lpos = heap_left(ipos);
   rpos = heap_right(ipos);

   if (lpos < heap_size(heap) && heap->compare(heap->tree[lpos], heap->
      tree[ipos]) > 0) {

      mpos = lpos;

      }

   else {

      mpos = ipos;

   }

   if (rpos < heap_size(heap) && heap->compare(heap->tree[rpos], heap->
      tree[mpos]) > 0) {

      mpos = rpos;

   }

   /**************************************************************************
   *                                                                         *
   *  Kiedyy mpos rwne jest ipos, drzewo znw stao si stert.             *
   *                                                                         *
   **************************************************************************/

   if (mpos == ipos) {

      break;

      }

   else {

      /***********************************************************************
      *                                                                      *
      *  Zamiana wartoci wza biecego z wybranym dzieckiem.              *
      *                                                                      *
      ***********************************************************************/

      temp = heap->tree[mpos];
      heap->tree[mpos] = heap->tree[ipos];
      heap->tree[ipos] = temp;

      /***********************************************************************
      *                                                                      *
      *  Przesunicie wza o jeden poziom w d.                            *
      *                                                                      *
      ***********************************************************************/

      ipos = mpos;

   }

}

return 0;

}
