/*************************************************************************
 *  Kompilacja:  javac FordFulkerson.java
 *  Wykonanie:    java FordFulkerson V E
 *  Zalenoci: FlowNetwork.java FlowEdge.java Queue.java
 *
 *  Algorytm Forda-Fulkersona do wyznaczania przepywu maksymalnego i
 *  przekroju minimalnego za pomoc reguy najkrtszej cieki powikszajcej.
 *
 *********************************************************************/

public class FordFulkerson {
    private boolean[] marked;     // marked[v] = prawda, jeli cieka s->v wystpuje w grafie rezydualnym
    private FlowEdge[] edgeTo;    // edgeTo[v] = ostatnia krawd na najkrtszej ciece rezydualnej s->v
    private double value;         // Bieca warto przepywu maksymalnego
  
    // Przepyw maksymalny z s do t w sieci przepywowej G 
    public FordFulkerson(FlowNetwork G, int s, int t) {
        value = excess(G, t);
        if (!isFeasible(G, s, t)) {
            throw new RuntimeException("Pocztkowy przepyw jest niemoliwy");
        }

        // Dopki istnieje cieka powikszajca, naley jej uywa
        while (hasAugmentingPath(G, s, t)) {

            // Obliczanie przepustowoci bdcej wskim gardem
            double bottle = Double.POSITIVE_INFINITY;
            for (int v = t; v != s; v = edgeTo[v].other(v)) {
                bottle = Math.min(bottle, edgeTo[v].residualCapacityTo(v));
            }

            // Powikszanie przepywu
            for (int v = t; v != s; v = edgeTo[v].other(v)) {
                edgeTo[v].addResidualFlowTo(v, bottle); 
            }

            value += bottle;
        }

        // Sprawdzanie warunkw optymalnoci
        assert check(G, s, t);
    }

    // Zwracanie wartoci przepywu maksymalnego
    public double value()  {
        return value;
    }

    // Czy v znajduje si po stronie s minimalnego przekroju s-t?
    public boolean inCut(int v)  {
        return marked[v];
    }


    // Zwraca ciek powikszajc, jeli istnieje (w przeciwnym razie zwraca null)
    private boolean hasAugmentingPath(FlowNetwork G, int s, int t) {
        edgeTo = new FlowEdge[G.V()];
        marked = new boolean[G.V()];

        // Przeszukiwanie wszerz
        Queue<Integer> q = new Queue<Integer>();
        q.enqueue(s);
        marked[s] = true;
        while (!q.isEmpty()) {
            int v = q.dequeue();

            for (FlowEdge e : G.adj(v)) {
                int w = e.other(v);
                
                if (e.residualCapacityTo(w) > 0) {
                    if (!marked[w]) {
                        edgeTo[w] = e;
                        marked[w] = true;
                        q.enqueue(w);
                    }
                }
            }
        }

        // Czy istnieje cieka powikszajca?
        return marked[t];
    }



    // Zwraca nadmiarowy przepyw w wierzchoku v
    private double excess(FlowNetwork G, int v) {
        double excess = 0.0;
        for (FlowEdge e : G.adj(v)) {
            if (v == e.from()) excess -= e.flow();
            else               excess += e.flow();
        }
        return excess;
    }

    private boolean isFeasible(FlowNetwork G, int s, int t) {
        double EPSILON = 1E-11;

        // Sprawdza, czy ograniczenia przepustowoci s spenione
        for (int v = 0; v < G.V(); v++) {
            for (FlowEdge e : G.adj(v)) {
                if (e.flow() < 0 || e.flow() > e.capacity()) {
                    System.err.println("Krawd nie spenia ogranicze przepustowoci: " + e);
                    return false;
                }
            }
        }

        // Sprawdza, czy przepyw netto w wierzchoku jest rwny zero (nie dotyczy to rda i ujcia)
        if (Math.abs(value + excess(G, s)) > EPSILON) {
            System.err.println("Nadmiar w rdle    = " + excess(G, s));
            System.err.println("Przepyw maksymalny = " + value);
            return false;
        }
        if (Math.abs(value - excess(G, t)) > EPSILON) {
            System.err.println("Nadmiar w ujciu    = " + excess(G, t));
            System.err.println("Przepyw maksymalny = " + value);
            return false;
        }
        for (int v = 0; v < G.V(); v++) {
            if (v == s || v == t) continue;
            else if (Math.abs(excess(G, v)) > EPSILON) {
                System.err.println("Przepyw netto w " + v + " nie jest rwny zero");
                return false;
            }
        }
        return true;
    }



    // Sprawdzanie warunkw optymalnoci
    private boolean check(FlowNetwork G, int s, int t) {

        // Sprawdzanie, czy przepyw jest moliwy
        if (!isFeasible(G, s, t)) {
            System.err.println("Przepyw jest niemoliwy");
            return false;
        }

        // Sprawdzanie, czy s znajduje si po stronie rda w przekroju minimalnym,
        //	a t nie znajduje si po tej stronie
        if (!inCut(s)) {
            System.err.println("rdo " + s + " nie znajduje si po stronie rda w przekroju minimalnym");
            return false;
        }
        if (inCut(t)) {
            System.err.println("ujcie " + t + " znajduje si po stroie rda w przekoju minimalnym");
            return false;
        }

        // Sprawdzanie, czy warto przekroju minimalnego = warto przekroju maksymalnego
        double mincutValue = 0.0;
        for (int v = 0; v < G.V(); v++) {
            for (FlowEdge e : G.adj(v)) {
                if ((v == e.from()) && inCut(e.from()) && !inCut(e.to()))
                    mincutValue += e.capacity();
            }
        }

        double EPSILON = 1E-11;
        if (Math.abs(mincutValue - value) > EPSILON) {
            System.err.println("Warto przekroju maksymalnego = " + value + ", warto przekroju minimalnego = " + mincutValue);
            return false;
        }

        return true;
    }


    // Klient testowy, ktry tworzy losow sie, rozwizuje problem
    //	przekroju maksymalnego i wywietla wyniki
    public static void main(String[] args) {

        // Tworzy sie przepywow o V wierzchokach i E krawdziach
        int V = Integer.parseInt(args[0]);
        int E = Integer.parseInt(args[1]);
        int s = 0, t = V-1;
        FlowNetwork G = new FlowNetwork(V, E);
        StdOut.println(G);

        // Oblicza przepyw maksymalny i przekrj minimalny
        FordFulkerson maxflow = new FordFulkerson(G, s, t);
        StdOut.println("Przepyw maksymalny z " + s + " do " + t);
        for (int v = 0; v < G.V(); v++) {
            for (FlowEdge e : G.adj(v)) {
                if ((v == e.from()) && e.flow() > 0)
                    StdOut.println("   " + e);
            }
        }

        // Wywietlanie przekroju minimalnego
        StdOut.print("Przekrj minimalny: ");
        for (int v = 0; v < G.V(); v++) {
            if (maxflow.inCut(v)) StdOut.print(v + " ");
        }
        StdOut.println();

        StdOut.println("Warto przepywu maksymalnego = " +  maxflow.value());
    }

}
