<?php
// Otwiera plik obsługi błęgów i konfigurację bazy danych.
require_once ('config.php');

// Klasa odpowiedzialna za prowadzenie weryfikacji AJAX lub PHP. 
class Validate
{
   // Przechowuje szczegóły połączenia z bazą danych.
   private $mMysqli;
   
   // Konstruktor otwierający połączenie z bazą danych.
   function __construct()
   {
      $this->mMysqli = new mysqli(DB_HOST, DB_USER, DB_PASSWORD, DB_DATABASE);
   }

   // Destruktor zamykający połącznie z bazą danych.
   function __destruct()
   {
      $this->mMysqli->close();         
   }
      
   // Wspomaga proces weryfikacji AJAX, sprawdza pojedynczą wartość.
   public function ValidateAJAX($inputValue, $fieldID)
   {
      // Sprawdza, które z pól jest weryfikowane i przeprowadza kontrolę jego zawartości.
      switch($fieldID)
      {
         // Sprawdza, czy nazwa użytkownika jest poprawna.
         case 'txtUsername':
            return $this->validateUserName($inputValue);
            break;
            
         // Sprawdza, czy nazwisko użytkownika jest poprawne.
         case 'txtName':
            return $this->validateName($inputValue);
            break;
            
         // Sprawdza, czy wybrano płeć.
         case 'selGender':
            return $this->validateGender($inputValue);
            break;
            
         // Sprawdza, czy dzień urodzenia ma poprawną wartość.
         case 'txtBthDay':
            return $this->validateBirthDay($inputValue);
            break;
         
         // Sprawdza, czy wybrano miesiąc urodzenia.
         case 'selBthMonth':
            return $this->validateBirthMonth($inputValue);
            break;
               
         // Sprawdza, czy rok urodzenia jest poprawny.
         case 'txtBthYear':
            return $this->validateBirthYear($inputValue);
            break;
            
         // Sprawdza format adresu e-mail. 
         case 'txtEmail':
            return $this->validateEmail($inputValue);
            break;
            
         // Sprawdza, czy numer telefonu został zapisany w poprawnym formacie. 
         case 'txtPhone':
            return $this->validatePhone($inputValue);
            break;
         
         // Sprawdza, czy potwierdzono znajomość warunków użytkowania.
         case 'chkReadTerms':
            return $this->validateReadTerms($inputValue);
            break;
      }
   }
   
   // Sprawdza poprawność wszystkich danych w chwili wysyłki.
   public function ValidatePHP()
   {
      // Flaga błędu - przyjmuje wartość 1, gdy zostanie napotkany błąd.
      $errorsExist = 0;
      // Czyści flagę sesji błędów.      
      if (isset($_SESSION['errors']))
         unset($_SESSION['errors']);
      // Domyślnie przyjmuje, że wszystkie pola zostały wypełnione poprawnie.
      $_SESSION['errors']['txtUsername'] = 'hidden';
      $_SESSION['errors']['txtName'] = 'hidden';
      $_SESSION['errors']['selGender'] = 'hidden';
      $_SESSION['errors']['txtBthDay'] = 'hidden';
      $_SESSION['errors']['selBthMonth'] = 'hidden';
      $_SESSION['errors']['txtBthYear'] = 'hidden';
      $_SESSION['errors']['txtEmail'] = 'hidden';
      $_SESSION['errors']['txtPhone'] = 'hidden';
      $_SESSION['errors']['chkReadTerms'] = 'hidden';
      
      // Weryfikuje poprawność nazwy użytkownika.
      if (!$this->validateUserName($_POST['txtUsername']))
      {
         $_SESSION['errors']['txtUsername'] = 'error';
         $errorsExist = 1;
      }
      
      // weryfikuje poprawność nazwiska.
      if (!$this->validateName($_POST['txtName']))
      {
         $_SESSION['errors']['txtName'] = 'error';
         $errorsExist = 1;
      }
   
      // Sprawdza, czy wybrano płeć.
      if (!$this->validateGender($_POST['selGender']))
      {
         $_SESSION['errors']['selGender'] = 'error';
         $errorsExist = 1;
      }
      
            // weryfikuje wartość dnia urodzenia.
      if (!$this->validateBirthDay($_POST['txtBthDay']))
      {
         $_SESSION['errors']['txtBthDay'] = 'error';
         $errorsExist = 1;
      }
      
      // Sprawdza, czy wybrano miesiąc urodzenia.
      if (!$this->validateBirthMonth($_POST['selBthMonth']))
      {
         $_SESSION['errors']['selBthMonth'] = 'error';
         $errorsExist = 1;
      }
      
      // Weryfikuje całość daty urodzenia.
      if (!$this->validateBirthYear($_POST['txtBthDay'] . '#' . 
                                                   $_POST['selBthMonth'] . '#' . 
                                                   $_POST['txtBthYear']))
      {
         $_SESSION['errors']['txtBthYear'] = 'error';
         $errorsExist = 1;
      }
      
      // Weryfikuje poprawność zapisu adresu e-mail.
      if (!$this->validateEmail($_POST['txtEmail']))
      {
         $_SESSION['errors']['txtEmail'] = 'error';
         $errorsExist = 1;
      }
   
      // Weryfikuje poprawność zapisu numeru telefonu.
      if (!$this->validatePhone($_POST['txtPhone']))
      {
         $_SESSION['errors']['txtPhone'] = 'error';
         $errorsExist = 1;
      }
      
      // Sprawdza, czy zaznaczono pole z potwierdzeniem znajomości warunków użytkowania.
      if (!isset($_POST['chkReadTerms']) || 
            !$this->validateReadTerms($_POST['chkReadTerms']))
      {
         $_SESSION['errors']['chkReadTerms'] = 'error';
         $_SESSION['values']['chkReadTerms'] = '';
         $errorsExist = 1;
      }

      // Jeśli nie wystąpiły błędy, zwraca nazwę strony potwierdzającej zakończenie weryfikacji.
      if ($errorsExist == 0)
      {
         return 'allok.php';
      }
      else
      {
         // Jeżeli wystąpiły błędy, zapisuje dane wprowadzone przez użytkownika.
         foreach ($_POST as $key => $value)
         {
            $_SESSION['values'][$key] = $_POST[$key];
         }
         return 'index.php';
      }
   }

   // Weryfikuje nazwę użytkownika (pole nie może być puste, a nazwa nie może pojawiać się już w bazie danych).
   private function validateUserName($value)
   {
      // trim and escape input value      
      $value = $this->mMysqli->real_escape_string(trim($value));
      // Użytkownik nie może pozostawić pustego pola.
      if ($value == null) 
         return 0; // Niepoprawne.
      // Sprawdza, czy nazwa użytkownika nie pojawia się w bazie.
      $query = $this->mMysqli->query('SELECT user_name FROM users ' .
                                                    'WHERE user_name="' . $value . '"');
      if ($this->mMysqli->affected_rows > 0)
         return '0'; // Niepoprawne.
      else
         return '1'; // Poprawne.
   }

   // Weryfikacja nazwiska użytkownika.
   private function validateName($value)
   {
      // trim and escape input value      
      $value = trim($value);
      // Użytkownik nie może pozostawić pustego pola.
      if ($value) 
         return 1; // Poprawne.
      else
         return 0; // Niepoprawne.
   }

   // Weryfikacja płci. 
   private function validateGender($value)
   {
      // Użytkownik musi podać płeć.
      return ($value == '0') ? 0 : 1;
   }   

      // Weryfikacja dnia urodzenia. 
   private function validateBirthDay($value)
   {
      // Dzień musi przyjmować wartości z przedziału od 1 do 31 i nie może być wartością pustą.   
      return ($value == '' || $value > 31 || $value < 1) ? 0 : 1;
   }
   
   // Weryfikacja miesiąca urodzenia.
   private function validateBirthMonth($value)
   {
      // Miesiąc nie może być wartością pustą i musi zawierać się w przedziale od 1 do 12.   
      return ($value == '' || $value > 12 || $value < 1) ? 0 : 1;
   }   
   
   // Weryfikuje rok urodzenia i całą datę.
   private function validateBirthYear($value)
   {
      // Rok musi zawierać się w przedziale od 1900 do 2000. 
      // Pobiera całą datę (dd#mm#rrrr).
      $date = explode('#', $value);
      // Data nie przechodzi weryfikacji, jeśli któreś z pól jest puste.   
      if (!$date[0]) return 0;
      if (!$date[1] || !is_numeric($date[1])) return 0;
      if (!$date[2] || !is_numeric($date[2])) return 0;
      // Sprawdza poprawność daty.
      return (checkdate($date[0], $date[1], $date[2])) ? 1 : 0;
   }

   // Weryfikuje poprawność adresu e-mail.
   private function validateEmail($value)
   {
      // Poprawne formaty adresu e-mail: *@*.*, *@*.*.*, *.*@*.*, *.*@*.*.*)
      return (!preg_match('/^[_a-z0-9-]+(\.[_a-z0-9-]+)*@[a-z0-9-]+(\.[a-z0-9-]+)*(\.[a-z]{2,3})$/i', $value)) ? 0 : 1;
   }
   
   // Weryfikuje numer telefonu.
   private function validatePhone($value)
   {
      // Poprawny format numeru telefonu: ###-##-## 
      return (!preg_match('/^[0-9]{3}-*[0-9]{2}-*[0-9]{2}$/', $value)) ? 0 : 1;
   }

   // Sprawdza, czy użytkownik potwierdził warunki użytkowania.
   private function validateReadTerms($value)
   {
      // Jeśli weryfikacja zakończy się powodzeniem, ustawia wartość zmiennej na 'true'.
      return ($value == 'true' || $value == 'on') ? 1 : 0;
   }
}
?>
